#!/bin/bash
set -euo pipefail 2>/dev/null || set -euo

# From wherever the script is stored, go to the project root where
# app/main.py lives and all the helpers (alembic, init_db, etc.) run.
PROJECT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
if [[ ! -f "$PROJECT_DIR/app/main.py" ]]; then
    PROJECT_DIR="$(cd "$PROJECT_DIR/.." && pwd)"
fi
cd "$PROJECT_DIR"

# Activate the virtual environment (fall back to `BAZARYA_VENV`, then active VIRTUAL_ENV, then `venv`)
if [[ -n "${BAZARYA_VENV:-}" ]]; then
    VENV_DIR="$BAZARYA_VENV"
elif [[ -n "${VIRTUAL_ENV:-}" ]]; then
    VENV_DIR="$VIRTUAL_ENV"
else
    VENV_DIR="$PROJECT_DIR/venv"
fi
if [[ ! -f "$VENV_DIR/bin/activate" ]]; then
    echo "Virtual environment not found at $VENV_DIR/bin/activate" >&2
    exit 1
fi
source "$VENV_DIR/bin/activate"

# Import env vars so alembic/init_db use the same credentials as the app.
if [[ -f .env ]]; then
    set -a
    source .env
    set +a
fi

# Touch the Passenger entry point so the shared host reloads the app.
if [[ -f "$PROJECT_DIR/passenger_wsgi.py" ]]; then
    touch "$PROJECT_DIR/passenger_wsgi.py"
fi

UVICORN_HOST="${UVICORN_HOST:-127.0.0.1}"
UVICORN_PORT="${UVICORN_PORT:-8080}"
UVICORN_LOG_DIR="logs/api_server"
UVICORN_CMD=(uvicorn app.main:app --host "$UVICORN_HOST" --port "$UVICORN_PORT")

echo "Stopping Bazarya-related python workers..."
for pattern in "python -m app.bots.hyper.main" "python -m app.bots.manager" "python -m app.bots.store" "uvicorn app.main:app"; do
    pkill -f "$pattern" || true
done
sleep 1

echo "Resetting database schema..."
python init_db.py
alembic upgrade head

echo "Starting hyper bot and process manager..."
mkdir -p logs/hyper_bot logs/store_bot
nohup python -m app.bots.hyper.main >> logs/hyper_bot/runner.log 2>&1 &
nohup python -m app.bots.manager >> logs/store_bot/runner.log 2>&1 &

echo "Starting Uvicorn on ${UVICORN_HOST}:${UVICORN_PORT}..."
mkdir -p "$UVICORN_LOG_DIR"
nohup "${UVICORN_CMD[@]}" >> "${UVICORN_LOG_DIR}/server.log" 2>&1 &

echo "Done: DB reset, immigration complete, bots running."
