#!/usr/bin/env python3
"""
Helper script برای مدیریت وبهوک تلگرام روی هاست اشتراکی.
مثال‌ها:
  python scripts/manage_webhook.py set --drop-pending
  python scripts/manage_webhook.py delete
  python scripts/manage_webhook.py info
"""

from __future__ import annotations

import argparse
import json
import os
from pathlib import Path
from typing import Any, Dict

import requests
from dotenv import load_dotenv

ProjectRoot = Path(__file__).resolve().parents[1]
EnvFile = ProjectRoot / ".env"


def _ensure_env_loaded() -> None:
    if EnvFile.exists():
        load_dotenv(EnvFile)
    else:
        load_dotenv()


def _require(name: str) -> str:
    value = os.getenv(name)
    if not value:
        raise RuntimeError(f"Environment variable {name} is required")
    return value


def _webhook_url() -> str:
    base = _require("WEBHOOK_BASE_URL").rstrip("/")
    secret = os.getenv("WEBHOOK_SECRET_PATH", "qv_Secret_9kP27").strip("/")
    return f"{base}/webhook/{secret}"


def _api(method: str) -> str:
    token = _require("BOT_TOKEN") or os.getenv("TELEGRAM_BOT_TOKEN")
    if not token:
        raise RuntimeError("BOT_TOKEN (or TELEGRAM_BOT_TOKEN) must be set")
    return f"https://api.telegram.org/bot{token}/{method}"


def call_api(method: str, payload: Dict[str, Any] | None = None, files=None) -> Dict[str, Any]:
    url = _api(method)
    response = requests.post(url, data=payload, files=files) if payload else requests.get(url)
    response.raise_for_status()
    data = response.json()
    print(json.dumps(data, ensure_ascii=False, indent=2))
    return data


def do_set(drop_pending: bool) -> None:
    url = _webhook_url()
    payload: Dict[str, Any] = {"url": url}
    if drop_pending:
        payload["drop_pending_updates"] = True
    secret_token = os.getenv("WEBHOOK_SECRET_TOKEN")
    if secret_token:
        payload["secret_token"] = secret_token
    cert_path = os.getenv("WEBHOOK_CERT_PATH")
    files = None
    if cert_path:
        cert_file = Path(cert_path)
        if not cert_file.is_file():
            raise RuntimeError(f"Certificate file {cert_file} not found")
        files = {"certificate": open(cert_file, "rb")}
    try:
        call_api("setWebhook", payload, files)
    finally:
        if files:
            files["certificate"].close()


def do_delete() -> None:
    call_api("deleteWebhook")


def do_info() -> None:
    call_api("getWebhookInfo")


def build_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(description="Manage Telegram webhook for QVerify bot")
    sub = parser.add_subparsers(dest="command", required=True)

    set_cmd = sub.add_parser("set", help="Set webhook to WEBHOOK_BASE_URL/WEBHOOK_SECRET_PATH")
    set_cmd.add_argument("--drop-pending", action="store_true", help="Drop pending updates on setWebhook")

    sub.add_parser("delete", help="Delete webhook")
    sub.add_parser("info", help="Get current webhook info")
    return parser


def main() -> None:
    _ensure_env_loaded()
    parser = build_parser()
    args = parser.parse_args()

    if args.command == "set":
        do_set(args.drop_pending)
    elif args.command == "delete":
        do_delete()
    elif args.command == "info":
        do_info()


if __name__ == "__main__":
    main()
