from fpdf import FPDF
import os, re
from pathlib import Path
import arabic_reshaper
from bidi.algorithm import get_display
from typing import Dict, Any, List

def _resolve_font_path() -> str:
    """
    Try env overrides first, then bundled assets.
    This keeps the bot portable across shared hosts.
    """
    base_dir = Path(__file__).resolve().parents[1]
    candidates = [
        os.environ.get("VAZIR_FONT_PATH"),
        os.environ.get("FONT_PATH"),
        base_dir / "public" / "fonts" / "Vazirmatn-Regular.ttf",
        Path(__file__).with_name("DejaVuSans.ttf"),
    ]
    for candidate in candidates:
        if not candidate:
            continue
        path = Path(candidate)
        if path.is_file():
            return str(path)
    raise FileNotFoundError(
        "Could not locate a Persian font. Set VAZIR_FONT_PATH or place "
        "Vazirmatn-Regular.ttf under public/fonts."
    )

FONT_PATH = _resolve_font_path()

# هر توکن بسیار بلند را قابل‌ شکست می‌کنیم:
def _soft_wrap_tokens(s: str, max_len: int = 25) -> str:
    # اگر رشته‌های بدون فاصله خیلی بلند باشند، بین‌شان فاصله می‌گذاریم
    def breaker(m):
        token = m.group(0)
        parts = [token[i:i+max_len] for i in range(0, len(token), max_len)]
        return " ".join(parts)
    # توکنِ بدون فاصله (حروف/عدد/علامت) طولانی
    return re.sub(r'[^\s]{%d,}' % max_len, breaker, s)

def fa(text: Any) -> str:
    if text is None: return ""
    s = str(text)
    s = _soft_wrap_tokens(s, max_len=25)
    # reshape + bidi برای راست‌به‌چپ
    return get_display(arabic_reshaper.reshape(s))

class FAPDF(FPDF):
    _font_ready = False
    def ensure_font(self):
        if not self._font_ready:
            self.add_font("Vazir", "", FONT_PATH, uni=True)
            self.set_font("Vazir", "", 12)
            # حاشیه‌ها و شکست خودکار صفحه
            self.set_margins(left=10, top=15, right=10)
            self.set_auto_page_break(auto=True, margin=15)
            self._font_ready = True
    def header(self):
        self.ensure_font()

def _safe_multicell(pdf: FAPDF, txt: str, line_h: float = 8, align: str = "R"):
    try:
        pdf.multi_cell(w=0, h=line_h, txt=txt, align=align)
    except Exception:
        # اگر به خاطر عرض/کلمه‌ی طولانی خطا داد، سایز فونت را موقتاً کم کن و دوباره
        size = pdf.font_size_pt
        try:
            pdf.set_font("Vazir", "", max(8, int(size * 0.85)))
            pdf.multi_cell(w=0, h=max(6, line_h*0.9), txt=txt, align=align)
        finally:
            pdf.set_font("Vazir", "", size)

def make_simple_pdf(lines: List[str], out_path: str):
    pdf = FAPDF()
    pdf.add_page()
    pdf.ensure_font()
    for line in lines:
        _safe_multicell(pdf, fa(line))
    pdf.output(out_path)
    return out_path

def generate_pdf(guarantee: Dict[str, Any], out_path: str):
    pdf = FAPDF()
    pdf.add_page()
    pdf.ensure_font()

    pdf.set_font("Vazir", "", 14)
    _safe_multicell(pdf, fa("گزارش ثبت گارانتی"))
    pdf.ln(2)
    pdf.set_font("Vazir", "", 12)

    shown = set()
    mapping = [
        ("نام فروشگاه", "store_name"),
        ("نام مشتری", "customer_name"),
        ("شماره موبایل", "mobile"),
        ("کد گارانتی", "code"),
        ("تاریخ", "date"),
    ]
    for label, key in mapping:
        val = guarantee.get(key)
        if val:
            _safe_multicell(pdf, fa(f"{label}: {val}"))
            shown.add(key)

    for k, v in guarantee.items():
        if k in shown or k in ("items", "products"): 
            continue
        if isinstance(v, (str, int, float)):
            _safe_multicell(pdf, fa(f"{k}: {v}"))

    items = guarantee.get("items") or guarantee.get("products")
    if isinstance(items, list) and items:
        pdf.ln(2)
        _safe_multicell(pdf, fa("اقلام:"))
        for i, it in enumerate(items, 1):
            if isinstance(it, dict):
                parts = []
                for kk, vv in it.items():
                    if isinstance(vv, (str, int, float)):
                        parts.append(f"{kk}: {vv}")
                line = " | ".join(parts) if parts else str(it)
            else:
                line = str(it)
            _safe_multicell(pdf, fa(f"{i}) {line}"))

    pdf.output(out_path)
    return out_path
