# -*- coding: utf-8 -*-
# ============== QVerify Bot (No-File Edition) ==============
# all comments are in Finglish; ui texts are Persian (formal)

import calendar
import os
import re
import unicodedata
import logging
import random
import string
from html import escape
from io import BytesIO
from datetime import datetime
from typing import Dict, List, Tuple
from pathlib import Path

from telegram import (
    Update, ReplyKeyboardMarkup, InlineKeyboardButton, InlineKeyboardMarkup, InputFile
)
from telegram.ext import (
    ApplicationBuilder, CommandHandler, MessageHandler,
    ConversationHandler, ContextTypes, filters, CallbackQueryHandler
)
from telegram.request import HTTPXRequest

from dotenv import load_dotenv
BASE_DIR = Path(__file__).resolve().parents[1]
load_dotenv(BASE_DIR / ".env")
import jdatetime
import pandas as pd

from db import (
    init_db, add_guarantee, get_by_code, search_guarantees,
    export_all, get_by_user, list_all,
    add_store, list_stores, search_stores, update_store,
    get_store_by_code,
    add_catalog, update_catalog, delete_catalog,
    list_catalogs, search_catalogs, get_catalog_by_code,
    list_catalog_files, replace_catalog_files, append_catalog_files,
)
# pdf_utils removed (no pdf generation)
# from pdf_utils import generate_pdf

# membership guard
from membership_guard import (
    require_membership, on_recheck_join,
    MAIN_MENU_USER as _MMU_JOIN, MAIN_MENU_ADMIN as _MMA_JOIN,
    CHANNEL_MEMBERSHIP_REQUIRED,
)

# ------------- helpers (regex/labels) -------------
def _norm(s: str) -> str:
    """normalize unicode to NFC to avoid ZWJ/half-space issues"""
    return unicodedata.normalize("NFC", s or "")

import re as _re
_MOBILE_PATTERN = _re.compile(r"^09\d{9}$")
_JALALI_DATE_PATTERN = _re.compile(r"^\s*(\d{4})[-/](\d{1,2})[-/](\d{1,2})\s*$")

def exact_label_pattern(label: str) -> _re.Pattern:
    """exact match regex with whitespace tolerance"""
    return _re.compile(rf"^\s*{_re.escape(_norm(label))}\s*$", flags=_re.UNICODE)

def _format_shamsi(dt: datetime) -> str:
    return jdatetime.datetime.fromgregorian(datetime=dt).strftime("%Y/%m/%d")

def _parse_shamsi_date(text: str) -> datetime | None:
    text = _norm(text or "")
    match = _JALALI_DATE_PATTERN.match(text)
    if not match:
        return None
    year, month, day = (int(p) for p in match.groups())
    try:
        greg = jdatetime.date(year, month, day).togregorian()
        return datetime(greg.year, greg.month, greg.day)
    except ValueError:
        return None

def _add_months(dt: datetime, months: int) -> datetime:
    total_month = dt.month - 1 + months
    year = dt.year + total_month // 12
    month = total_month % 12 + 1
    day = min(dt.day, calendar.monthrange(year, month)[1])
    return dt.replace(year=year, month=month, day=day)

def _build_warranty_details(name: str, mobile: str, store_code: str, code: str, start_dt: datetime, end_dt: datetime) -> str:
    start = _format_shamsi(start_dt)
    end = _format_shamsi(end_dt)
    store_label = _format_store_label(store_code)
    return (
        f"🆔 کد گارانتی: <code>{escape(code)}</code>\n"
        f"🔰 نام مشتری: {escape(name)}\n"
        f"📞 موبایل: {escape(mobile)}\n"
        f"🏪 فروشگاه: {store_label}\n"
        f"🗓 تاریخ شروع: {start}\n"
        f"🗓 تاریخ پایان: {end}"
    )


def _normalize_store_code(text: str) -> str:
    return (text or "").strip().upper()


def _format_store_label(code: str) -> str:
    normalized = _normalize_store_code(code)
    if not normalized:
        return "نامشخص"
    store = get_store_by_code(normalized)
    if store:
        return f"{escape(store[1])} ({escape(store[0])})"
    return escape(normalized)


def _render_store_lines(stores: List[Tuple[str, str]]) -> str:
    if not stores:
        return "هیچ فروشگاهی ثبت نشده است."
    lines: List[str] = []
    for idx, (code, name) in enumerate(stores, start=1):
        lines.append(f"{idx}. {escape(name)} — <code>{escape(code)}</code>")
    return "\n".join(lines)

def _normalize_catalog_code(text: str) -> str:
    return (text or "").strip().upper()

def _render_catalog_lines(catalogs: List[Tuple[str, str, int]]) -> str:
    if not catalogs:
        return "کاتالوگی پیدا نشد."
    lines: List[str] = []
    for idx, (code, title, file_count) in enumerate(catalogs, start=1):
        suffix = f"({file_count} فایل)" if file_count is not None else ""
        lines.append(f"{idx}. {escape(title)} — <code>{escape(code)}</code> {suffix}".strip())
    return "\n".join(lines)

def _copy_code_markup(code: str) -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        [[InlineKeyboardButton(text="کپی کد گارانتی", switch_inline_query_current_chat=code)]]
    )

# ui labels (use the same constants everywhere)
LBL_NEW            = "ثبت گارانتی جدید"
LBL_MINE           = "گارانتی های من"
LBL_SEARCH         = "جست و جو"
LBL_LIST_RECENTS   = "لیست آخرین ها"
LBL_EXPORT_XLSX    = "خروجی اکسل"
LBL_HELP           = "راهنما"
LBL_STORE_SEARCH   = "کد فروشگاه"
LBL_MANAGE_STORES  = "مدیریت فروشگاه ها"
LBL_STORE_ADD      = "ثبت فروشگاه جدید"
LBL_STORE_EDIT     = "ویرایش فروشگاه"
LBL_STORE_LIST     = "لیست فروشگاه ها"
LBL_CATALOG_SEARCH = "کاتالوگ کالا"
LBL_MANAGE_CATALOGS = "مدیریت کاتالوگ ها"
LBL_CATALOG_ADD    = "ثبت کاتالوگ"
LBL_CATALOG_EDIT   = "ویرایش کاتالوگ"
LBL_CATALOG_DELETE = "حذف کاتالوگ"
LBL_CATALOG_LIST   = "لیست کاتالوگ ها"
LBL_UPLOAD_DONE    = "پایان بارگذاری"
LBL_KEEP_FILES     = "حفظ فایل های فعلی"
LBL_BACK           = "Back to Menu"

# ------------- logging (stdout only; no files) -------------
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s [%(levelname)s] %(name)s - %(message)s",
)
logger = logging.getLogger("qverify-bot")

# ------------- env/load -------------
# (dotenv is loaded near the top before other imports)

# force clean network (ignore inherited proxies)
for proxy_var in ("HTTP_PROXY", "HTTPS_PROXY", "http_proxy", "https_proxy"):
    os.environ.pop(proxy_var, None)

def _get_env_list(name: str) -> List[int]:
    """parse comma-separated int ids from env"""
    raw_value = os.getenv(name, "")
    items = [i.strip() for i in raw_value.split(",") if i.strip()]
    if not items:
        raise RuntimeError(f"Environment variable {name} must contain at least one value")
    try:
        return [int(i) for i in items]
    except ValueError as exc:
        raise RuntimeError(f"Environment variable {name} must contain numeric IDs") from exc

# prefer BOT_TOKEN; fall back to TELEGRAM_BOT_TOKEN
TOKEN = os.getenv("BOT_TOKEN") or os.getenv("TELEGRAM_BOT_TOKEN")
if not TOKEN:
    raise RuntimeError("BOT_TOKEN (or TELEGRAM_BOT_TOKEN) environment variable is required")

ADMIN_IDS: List[int] = _get_env_list("ADMIN_IDS")

# ------------- conversation states -------------
(PHOTO, INVOICE_DATE, STORE_CODE, NAME, MOBILE, SEARCH_GUARANTEE, STORE_SEARCH,
 STORE_ADMIN_MENU, STORE_ADMIN_ADD_NAME, STORE_ADMIN_ADD_CODE, STORE_ADMIN_EDIT_SELECT, STORE_ADMIN_EDIT_NAME, STORE_ADMIN_EDIT_CODE,
 CATALOG_SEARCH, CATALOG_ADMIN_MENU, CATALOG_ADMIN_ADD_TITLE, CATALOG_ADMIN_ADD_CODE, CATALOG_ADMIN_ADD_FILES,
 CATALOG_ADMIN_EDIT_SELECT, CATALOG_ADMIN_EDIT_TITLE, CATALOG_ADMIN_EDIT_CODE, CATALOG_ADMIN_EDIT_FILES, CATALOG_ADMIN_DELETE_CONFIRM) = range(23)

# in-memory user scratch (ephemeral)
user_data: Dict[int, Dict] = {}

# ------------- utilities -------------
def generate_code() -> str:
    """generate human friendly code"""
    return "GRNT-" + ''.join(random.choices(string.digits, k=6))

def is_admin(user_id: int) -> bool:
    return user_id in ADMIN_IDS

def is_back_text(text: str) -> bool:
    """detect back/menu intents"""
    t = _norm((text or "").strip())
    return t in (LBL_BACK, "بازگشت به منو", "بازگشت", "منو") or t.lower() in ("/menu", "/start")

# ------------- keyboards -------------
MAIN_MENU_USER = ReplyKeyboardMarkup(
    [[LBL_NEW, LBL_MINE],
     [LBL_STORE_SEARCH, LBL_CATALOG_SEARCH]],
    resize_keyboard=True,
    is_persistent=True
)
MAIN_MENU_ADMIN = ReplyKeyboardMarkup(
    [[LBL_NEW, LBL_MINE],
     [LBL_SEARCH, LBL_LIST_RECENTS],
     [LBL_STORE_SEARCH, LBL_CATALOG_SEARCH],
     [LBL_MANAGE_STORES, LBL_MANAGE_CATALOGS],
     [LBL_EXPORT_XLSX, LBL_HELP]],
    resize_keyboard=True,
    is_persistent=True
)
CONV_MENU = ReplyKeyboardMarkup([[LBL_BACK]], resize_keyboard=True, is_persistent=True)
STORE_MANAGE_MENU = ReplyKeyboardMarkup(
    [[LBL_STORE_ADD, LBL_STORE_EDIT],
     [LBL_STORE_LIST, LBL_BACK]],
    resize_keyboard=True,
    is_persistent=True
)
CATALOG_MANAGE_MENU = ReplyKeyboardMarkup(
    [[LBL_CATALOG_ADD, LBL_CATALOG_EDIT],
     [LBL_CATALOG_DELETE, LBL_CATALOG_LIST],
     [LBL_BACK]],
    resize_keyboard=True,
    is_persistent=True
)
CATALOG_FILES_MENU = ReplyKeyboardMarkup(
    [[LBL_UPLOAD_DONE],
     [LBL_BACK]],
    resize_keyboard=True,
    is_persistent=True
)
CATALOG_FILES_EDIT_MENU = ReplyKeyboardMarkup(
    [[LBL_UPLOAD_DONE, LBL_KEEP_FILES],
     [LBL_BACK]],
    resize_keyboard=True,
    is_persistent=True
)

# ------------- handlers -------------
@require_membership(is_admin)
async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """start/menu entry"""
    uid = update.message.from_user.id
    kb = MAIN_MENU_ADMIN if is_admin(uid) else MAIN_MENU_USER
    await update.message.reply_text(
        "به ربات ثبت و مدیریت گارانتی خوش آمدید. لطفا یکی از گزینه های زیر را انتخاب کنید.",
        reply_markup=kb,
    )
    return ConversationHandler.END

async def cancel(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """end current flow and return to main menu"""
    uid = update.message.from_user.id
    kb = MAIN_MENU_ADMIN if is_admin(uid) else MAIN_MENU_USER
    await update.message.reply_text("به منوی اصلی بازگشتید.", reply_markup=kb)
    user_data.pop(uid, None)
    try:
        _reset_store_admin_context(context)
        _reset_catalog_admin_context(context)
    except Exception:
        pass
    return ConversationHandler.END

@require_membership(is_admin)
async def start_new(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """begin new guarantee flow"""
    await update.message.reply_text("لطفا تصویر فاکتور خرید را ارسال کنید.", reply_markup=CONV_MENU)
    return PHOTO

async def receive_photo(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """receive invoice photo (no disk save; keep file_id only)"""
    photo_size = update.message.photo[-1]
    # no download_to_drive -> we only keep file_id for re-send and DB
    user_data[update.message.from_user.id] = {
        "photo_file_id": photo_size.file_id,
        "photo_message_id": update.message.message_id,
    }
    await update.message.reply_text(
        "تاریخ فاکتور خرید را به صورت شمسی وارد کنید (مثال: 1404/08/01).",
        reply_markup=CONV_MENU,
    )
    return INVOICE_DATE


async def receive_invoice_date(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """invoice date step (Shamsi)"""
    if is_back_text(update.message.text):
        return await cancel(update, context)
    parsed = _parse_shamsi_date(update.message.text)
    if not parsed:
        await update.message.reply_text(
            "فرمت تاریخ صحیح نیست. لطفا مانند 1404/08/01 وارد کنید.",
            reply_markup=CONV_MENU,
        )
        return INVOICE_DATE
    uid = update.message.from_user.id
    user_data.setdefault(uid, {})
    user_data[uid]["invoice_date"] = parsed
    await update.message.reply_text("کد فروشگاه را وارد کنید.", reply_markup=CONV_MENU)
    return STORE_CODE

async def receive_store(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """store name step"""
    if is_back_text(update.message.text):
        return await cancel(update, context)
    user_data.setdefault(update.message.from_user.id, {})
    store_code = _normalize_store_code(update.message.text)
    if not store_code:
        await update.message.reply_text(
            "کد فروشگاه وارد نشده است. اگر کد را به خاطر ندارید از دکمه «کد فروشگاه» استفاده کنید.",
            reply_markup=CONV_MENU,
        )
        return STORE_CODE
    if not get_store_by_code(store_code):
        await update.message.reply_text(
            "کد فروشگاه معتبر نیست. لطفا از لیست فروشگاه ها کد صحیح را پیدا کرده و مجددا ارسال کنید.",
            reply_markup=CONV_MENU,
        )
        return STORE_CODE
    user_data[update.message.from_user.id]["store_code"] = store_code
    await update.message.reply_text("نام مشتری را وارد کنید.", reply_markup=CONV_MENU)
    return NAME

async def receive_name(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """customer name step"""
    if is_back_text(update.message.text):
        return await cancel(update, context)
    user_data[update.message.from_user.id]["name"] = update.message.text.strip()
    await update.message.reply_text("شماره موبایل را وارد کنید.", reply_markup=CONV_MENU)
    return MOBILE

async def broadcast_to_admins(
    context: ContextTypes.DEFAULT_TYPE, *,
    uid: int, caption: str, photo_file_id: str, copy_message_id: int
) -> List[Tuple[int, str]]:
    """notify admins with photo; fallback to copy_message on failure (no files)"""
    failures: List[Tuple[int, str]] = []
    for admin_id in ADMIN_IDS:
        try:
            await context.bot.send_photo(
                chat_id=admin_id,
                photo=photo_file_id,
                caption=caption,
                parse_mode="HTML",
            )
        except Exception as e:
            try:
                await context.bot.copy_message(chat_id=admin_id, from_chat_id=uid, message_id=copy_message_id)
            except Exception as e2:
                failures.append((admin_id, f"{e}"))
                logging.warning("notify admin %s failed: %s | fallback: %s", admin_id, e, e2)
    return failures

async def receive_mobile(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """mobile step -> persist and finalize (no pdf, no disk)"""
    if is_back_text(update.message.text):
        return await cancel(update, context)

    uid = update.message.from_user.id
    mobile = (update.message.text or "").strip()

    if not _MOBILE_PATTERN.match(mobile):
        await update.message.reply_text(
            "شماره موبایل معتبر نیست. باید ۱۱ رقمی و با 09 شروع شود.",
            reply_markup=CONV_MENU,
        )
        return MOBILE

    data = user_data.get(uid, {})
    issued_at = data.get("invoice_date") or datetime.now()
    expiry = _add_months(issued_at, 18)

    # unique code generation
    while True:
        code = generate_code()
        if not get_by_code(code):
            break

    # NOTE: 'photo' field stores Telegram file_id (no disk path)
    store_code = data.get("store_code", "")
    guarantee = {
        "code": code,
        "user_id": uid,
        "name": data.get("name", ""),
        "mobile": mobile,
        "store": store_code,
        "photo": data.get("photo_file_id", ""),   # keep file_id
        "issued_at": issued_at.isoformat(),
        "expires_at": expiry.isoformat(),
    }
    add_guarantee(guarantee)

    # pdf removed: no generate_pdf, no send_document
    details = _build_warranty_details(
        data.get("name", ""),
        mobile,
        store_code,
        code,
        issued_at,
        expiry,
    )
    await update.message.reply_text(
        "✨ گارانتی شما با موفقیت ثبت شد ✅\n"
        "اطلاعات کامل گارانتی:\n"
        f"{details}",
        reply_markup=_copy_code_markup(code),
        parse_mode="HTML",
    )

    caption = f"درخواست جدید ثبت گارانتی:\n{details}"
    failures = await broadcast_to_admins(
        context,
        uid=uid,
        caption=caption,
        photo_file_id=data.get("photo_file_id", ""),
        copy_message_id=data.get("photo_message_id", 0),
    )

    user_data.pop(uid, None)
    kb = MAIN_MENU_ADMIN if is_admin(uid) else MAIN_MENU_USER
    await update.message.reply_text("برای ادامه لطفا یکی از گزینه های زیر را انتخاب کنید.", reply_markup=kb)

    if failures and ADMIN_IDS:
        note = "اطلاع رسانی به برخی از ادمین ها ناموفق بود:\n" + "\n".join([f"{aid}: {err}" for aid, err in failures])
        try:
            await context.bot.send_message(chat_id=ADMIN_IDS[0], text=note)
        except Exception:
            pass

    return ConversationHandler.END

@require_membership(is_admin, admin_only=True)
async def check(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """/check <code>"""
    if not context.args:
        await update.message.reply_text("لطفا کد گارانتی را ارسال کنید.")
        return
    code = context.args[0]
    result = get_by_code(code)
    if not result:
        await update.message.reply_text("هیچ گارانتی با این کد یافت نشد.")
        return
    _, _, name, mobile, store, _, issued_at, expires_at = result
    remaining = (datetime.fromisoformat(expires_at) - datetime.now()).days
    j_expires = jdatetime.datetime.fromgregorian(datetime=datetime.fromisoformat(expires_at)).strftime("%Y/%m/%d")
    store_label = _format_store_label(store)
    await update.message.reply_text(
        f"کد: {code}\n"
        f"نام مشتری: {name}\n"
        f"شماره موبایل: {mobile}\n"
        f"فروشگاه: {store_label}\n"
        f"تاریخ انقضا: {j_expires} (باقی مانده: {remaining} روز)"
    )

@require_membership(is_admin, admin_only=True)
async def search_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """start textual search flow"""
    await update.message.reply_text("عبارت مورد نظر برای جست و جو را ارسال کنید.", reply_markup=CONV_MENU)
    return SEARCH_GUARANTEE

@require_membership(is_admin, admin_only=True)
async def search_receive(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """receive query and show results"""
    if is_back_text(update.message.text):
        return await cancel(update, context)
    query = _norm((update.message.text or "").strip())
    if not query:
        await update.message.reply_text("عبارت جست و جو خالی است.")
        return SEARCH_GUARANTEE
    results = search_guarantees(query)
    if not results:
        await update.message.reply_text("نتیجه ای یافت نشد.")
    else:
        chat_id = update.effective_chat.id
        for row in results[:10]:
            issued_at = datetime.fromisoformat(row[6])
            expires_at = datetime.fromisoformat(row[7])
            details = _build_warranty_details(
                row[2], row[3], row[4], row[0], issued_at, expires_at
            )
            caption = f"اطلاعات گارانتی:\n{details}"
            if row[5]:
                try:
                    await context.bot.send_photo(
                        chat_id=chat_id,
                        photo=row[5],
                        caption=caption,
                        parse_mode="HTML",
                    )
                    continue
                except Exception:
                    pass
            await update.message.reply_text(caption, parse_mode="HTML")
    return await cancel(update, context)


@require_membership(is_admin)
async def store_search_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """allow users to search store codes by name"""
    await update.message.reply_text(
        "نام یا بخشی از نام فروشگاه را ارسال کنید تا کدهای مرتبط نشان داده شوند.",
        reply_markup=CONV_MENU,
    )
    return STORE_SEARCH


@require_membership(is_admin)
async def store_search_receive(update: Update, context: ContextTypes.DEFAULT_TYPE):
    if is_back_text(update.message.text):
        return await cancel(update, context)
    query = _norm((update.message.text or "").strip())
    if not query:
        await update.message.reply_text("لطفا بخشی از نام یا کد را ارسال کنید.", reply_markup=CONV_MENU)
        return STORE_SEARCH
    results = search_stores(query)
    if not results:
        await update.message.reply_text("هیچ فروشگاهی یافت نشد.", reply_markup=CONV_MENU)
        return await cancel(update, context)
    lines = _render_store_lines(results)
    user = update.effective_user
    menu_kb = MAIN_MENU_ADMIN if user and is_admin(user.id) else MAIN_MENU_USER
    await update.message.reply_text(
        "نتایج جست و جو:\n" + lines + "\n\nکد مورد نظر را در مرحله ثبت وارد کنید.",
        reply_markup=menu_kb,
        parse_mode="HTML",
    )
    return ConversationHandler.END


@require_membership(is_admin)
async def catalog_send_files_callback(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """send catalog files when user clicks on an inline button"""
    query = update.callback_query
    if not query or not query.data:
        return

    data = str(query.data)
    if not data.startswith("catalog:"):
        return

    code = _normalize_catalog_code(data.split(":", 1)[1])
    await query.answer()

    catalog = get_catalog_by_code(code)
    files = list_catalog_files(code)
    user = query.from_user
    menu_kb = MAIN_MENU_ADMIN if user and is_admin(user.id) else MAIN_MENU_USER

    if catalog:
        header = f"{escape(catalog[1])} — <code>{escape(catalog[0])}</code>"
    else:
        header = f"<code>{escape(code)}</code>"

    if not files:
        await query.message.reply_text(
            header + "\nفایلی برای این کاتالوگ ثبت نشده است.",
            parse_mode="HTML",
            reply_markup=menu_kb,
        )
        return

    first_caption = header + f"\nتعداد فایل: {len(files)}"
    first = True
    for file_id, file_name in files:
        caption = first_caption if first else (file_name or "")
        first = False
        try:
            await query.message.reply_document(
                document=file_id,
                caption=caption or None,
                parse_mode="HTML" if caption else None,
                reply_markup=menu_kb if caption else None,
            )
        except Exception:
            await query.message.reply_text(
                header + "\nدریافت فایل ناموفق بود.",
                parse_mode="HTML",
                reply_markup=menu_kb,
            )
            break

@require_membership(is_admin)
async def catalog_search_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """search catalogs by code/title and show as buttons"""
    await update.message.reply_text(
        "کد یا عنوان کالا را ارسال کنید تا لیست کاتالوگ‌های مرتبط به‌صورت دکمه نمایش داده شود.",
        reply_markup=CONV_MENU,
    )
    return CATALOG_SEARCH


@require_membership(is_admin)
async def catalog_search_receive(update: Update, context: ContextTypes.DEFAULT_TYPE):
    if is_back_text(update.message.text):
        return await cancel(update, context)
    query = _norm((update.message.text or "").strip())
    if not query:
        await update.message.reply_text("لطفاً عبارت جستجو را وارد کنید.", reply_markup=CONV_MENU)
        return CATALOG_SEARCH

    results = search_catalogs(query)
    if not results:
        await update.message.reply_text("کاتالوگی پیدا نشد.", reply_markup=CONV_MENU)
        return await cancel(update, context)

    user = update.effective_user
    limited = results[:10]
    keyboard_rows: List[List[InlineKeyboardButton]] = []
    for code, title, file_count in limited:
        label_parts = [title or code]
        if code:
            label_parts.append(f"({code})")
        if file_count:
            label_parts.append(f"[{file_count} فایل]")
        text = " ".join(label_parts)
        keyboard_rows.append(
            [InlineKeyboardButton(text=text, callback_data=f"catalog:{code}")]
        )

    extra_note = ""
    if len(results) > len(limited):
        extra_note = "\n(فقط چند مورد اول نمایش داده شد.)"

    await update.message.reply_text(
        "نتایج یافت شد؛ برای دریافت فایل کاتالوگ روی یکی از گزینه‌ها کلیک کنید." + extra_note,
        reply_markup=InlineKeyboardMarkup(keyboard_rows),
    )
    return ConversationHandler.END




def _reset_store_admin_context(context: ContextTypes.DEFAULT_TYPE) -> None:
    for key in ("store_temp_name", "store_edit_target", "store_edit_original_name", "store_edit_new_name"):
        context.user_data.pop(key, None)

def _reset_catalog_admin_context(context: ContextTypes.DEFAULT_TYPE) -> None:
    for key in (
        "catalog_new_title",
        "catalog_new_code",
        "catalog_new_files",
        "catalog_edit_target",
        "catalog_edit_original_title",
        "catalog_edit_new_title",
        "catalog_edit_new_code",
        "catalog_edit_files",
        "catalog_edit_new_files",
    ):
        context.user_data.pop(key, None)


@require_membership(is_admin, admin_only=True)
async def store_manage_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    await update.message.reply_text(
        "کدام عملیات را می خواهید انجام دهید؟",
        reply_markup=STORE_MANAGE_MENU,
    )
    _reset_store_admin_context(context)
    return STORE_ADMIN_MENU


@require_membership(is_admin, admin_only=True)
async def store_manage_action(update: Update, context: ContextTypes.DEFAULT_TYPE):
    text = _norm((update.message.text or "").strip())
    if text == _norm(LBL_STORE_ADD):
        await update.message.reply_text("نام فروشگاه را وارد کنید.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_ADD_NAME
    if text == _norm(LBL_STORE_EDIT):
        await update.message.reply_text("کد فروشگاه مورد نظر را وارد کنید.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_EDIT_SELECT
    if text == _norm(LBL_STORE_LIST):
        stores = list_stores()
        lines = _render_store_lines(stores)
        await update.message.reply_text(
            "فروشگاه های ثبت شده:\n" + lines,
            reply_markup=STORE_MANAGE_MENU,
            parse_mode="HTML",
        )
        return STORE_ADMIN_MENU
    await update.message.reply_text(
        "گزینه ای از لیست انتخاب نشده است. لطفا یکی از دکمه های بالا را لمس کنید.",
        reply_markup=STORE_MANAGE_MENU,
    )
    return STORE_ADMIN_MENU


@require_membership(is_admin, admin_only=True)
async def store_admin_add_name(update: Update, context: ContextTypes.DEFAULT_TYPE):
    name = (update.message.text or "").strip()
    if not name:
        await update.message.reply_text("نام فروشگاه نمی تواند خالی باشد.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_ADD_NAME
    context.user_data["store_temp_name"] = name
    await update.message.reply_text("کد یکتای فروشگاه را وارد کنید.", reply_markup=STORE_MANAGE_MENU)
    return STORE_ADMIN_ADD_CODE


@require_membership(is_admin, admin_only=True)
async def store_admin_add_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    raw_code = update.message.text or ""
    code = _normalize_store_code(raw_code)
    if not code:
        await update.message.reply_text("کد فروشگاه معتبر نیست.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_ADD_CODE
    if get_store_by_code(code):
        await update.message.reply_text("کد وارد شده قبلا ثبت شده است.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_ADD_CODE
    name = context.user_data.get("store_temp_name", "")
    try:
        add_store(code, name)
    except ValueError as exc:
        await update.message.reply_text(str(exc), reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_ADD_CODE
    except Exception:
        await update.message.reply_text("کد وارد شده قبلا ثبت شده است.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_ADD_CODE
    _reset_store_admin_context(context)
    await update.message.reply_text(
        f"فروشگاه «{escape(name)}» با کد <code>{escape(code)}</code> ثبت شد.",
        reply_markup=MAIN_MENU_ADMIN,
        parse_mode="HTML",
    )
    return ConversationHandler.END


@require_membership(is_admin, admin_only=True)
async def store_admin_edit_select(update: Update, context: ContextTypes.DEFAULT_TYPE):
    raw_code = update.message.text or ""
    code = _normalize_store_code(raw_code)
    store = get_store_by_code(code)
    if not store:
        await update.message.reply_text("فروشگاهی با این کد یافت نشد.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_EDIT_SELECT
    context.user_data["store_edit_target"] = store[0]
    context.user_data["store_edit_original_name"] = store[1]
    await update.message.reply_text(
        f"نام فعلی فروشگاه: «{escape(store[1])}». نام جدید را وارد کنید (برای حفظ نام فعلی همان را ارسال کنید).",
        reply_markup=STORE_MANAGE_MENU,
    )
    return STORE_ADMIN_EDIT_NAME


@require_membership(is_admin, admin_only=True)
async def store_admin_edit_name(update: Update, context: ContextTypes.DEFAULT_TYPE):
    name = (update.message.text or "").strip()
    if not name:
        name = context.user_data.get("store_edit_original_name", "")
    context.user_data["store_edit_new_name"] = name
    await update.message.reply_text(
        "کد جدید فروشگاه را وارد کنید (اگر تغییر نمی دهید همان کد فعلی را ارسال کنید).",
        reply_markup=STORE_MANAGE_MENU,
    )
    return STORE_ADMIN_EDIT_CODE


@require_membership(is_admin, admin_only=True)
async def store_admin_edit_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    target_code = context.user_data.get("store_edit_target")
    if not target_code:
        return await cancel(update, context)
    raw_code = update.message.text or ""
    new_code = _normalize_store_code(raw_code)
    if not new_code:
        await update.message.reply_text("کد فروشگاه معتبر نیست.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_EDIT_CODE
    existing = get_store_by_code(new_code)
    if existing and existing[0] != target_code:
        await update.message.reply_text("این کد قبلا توسط فروشگاه دیگری استفاده شده است.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_EDIT_CODE
    new_name = context.user_data.get("store_edit_new_name") or context.user_data.get("store_edit_original_name", "")
    try:
        updated = update_store(target_code, new_name=new_name, new_code=new_code)
    except Exception:
        await update.message.reply_text("این کد قبلا ثبت شده است.", reply_markup=STORE_MANAGE_MENU)
        return STORE_ADMIN_EDIT_CODE
    _reset_store_admin_context(context)
    if not updated:
        await update.message.reply_text("هیچ تغییری ثبت نشد.", reply_markup=MAIN_MENU_ADMIN)
        return ConversationHandler.END
    await update.message.reply_text(
        f"فروشگاه با موفقیت به روزرسانی شد: «{escape(new_name)}» — <code>{escape(new_code)}</code>",
        reply_markup=MAIN_MENU_ADMIN,
        parse_mode="HTML",
    )
    return ConversationHandler.END


@require_membership(is_admin, admin_only=True)
async def catalog_manage_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    await update.message.reply_text(
        "کدام عملیات روی کاتالوگ‌ها را انجام می‌دهید؟",
        reply_markup=CATALOG_MANAGE_MENU,
    )
    _reset_catalog_admin_context(context)
    return CATALOG_ADMIN_MENU


@require_membership(is_admin, admin_only=True)
async def catalog_manage_action(update: Update, context: ContextTypes.DEFAULT_TYPE):
    text = _norm((update.message.text or "").strip())
    if text == _norm(LBL_CATALOG_ADD):
        _reset_catalog_admin_context(context)
        await update.message.reply_text("عنوان کالا را وارد کنید.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_ADD_TITLE
    if text == _norm(LBL_CATALOG_EDIT):
        await update.message.reply_text("کد کاتالوگ مورد نظر را ارسال کنید.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_EDIT_SELECT
    if text == _norm(LBL_CATALOG_DELETE):
        await update.message.reply_text("کد کالایی که باید حذف شود را وارد کنید.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_DELETE_CONFIRM
    if text == _norm(LBL_CATALOG_LIST):
        catalogs = list_catalogs()
        lines = _render_catalog_lines(catalogs)
        await update.message.reply_text(
            "کاتالوگ‌های ثبت شده:\n" + lines,
            reply_markup=CATALOG_MANAGE_MENU,
            parse_mode="HTML",
        )
        return CATALOG_ADMIN_MENU
    await update.message.reply_text(
        "گزینه‌ای از لیست انتخاب نشده است. لطفاً یکی از دکمه‌های بالا را لمس کنید.",
        reply_markup=CATALOG_MANAGE_MENU,
    )
    return CATALOG_ADMIN_MENU


@require_membership(is_admin, admin_only=True)
async def catalog_admin_add_title(update: Update, context: ContextTypes.DEFAULT_TYPE):
    title = (update.message.text or "").strip()
    if not title:
        await update.message.reply_text("عنوان کالا نمی‌تواند خالی باشد.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_ADD_TITLE
    context.user_data["catalog_new_title"] = title
    await update.message.reply_text("کد کالا را وارد کنید.", reply_markup=CATALOG_MANAGE_MENU)
    return CATALOG_ADMIN_ADD_CODE


@require_membership(is_admin, admin_only=True)
async def catalog_admin_add_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    raw_code = update.message.text or ""
    code = _normalize_catalog_code(raw_code)
    if not code:
        await update.message.reply_text("کد کالا معتبر نیست.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_ADD_CODE
    if get_catalog_by_code(code):
        await update.message.reply_text("برای این کد قبلاً کاتالوگ ثبت شده است.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_ADD_CODE
    context.user_data["catalog_new_code"] = code
    context.user_data["catalog_new_files"] = []
    await update.message.reply_text(
        "فایل یا فایل‌های کاتالوگ را ارسال کنید. پس از اتمام «پایان بارگذاری» را بزنید.",
        reply_markup=CATALOG_FILES_MENU,
    )
    return CATALOG_ADMIN_ADD_FILES


@require_membership(is_admin, admin_only=True)
async def catalog_admin_add_files(update: Update, context: ContextTypes.DEFAULT_TYPE):
    text = _norm((update.message.text or "").strip())
    if is_back_text(update.message.text):
        _reset_catalog_admin_context(context)
        return await cancel(update, context)
    files_list = context.user_data.setdefault("catalog_new_files", [])
    if text == _norm(LBL_UPLOAD_DONE):
        if not files_list:
            await update.message.reply_text("حداقل یک فایل کاتالوگ بارگذاری کنید.", reply_markup=CATALOG_FILES_MENU)
            return CATALOG_ADMIN_ADD_FILES
        title = context.user_data.get("catalog_new_title", "")
        code = context.user_data.get("catalog_new_code", "")
        try:
            add_catalog(code, title)
            append_catalog_files(code, files_list)
        except (ValueError, Exception) as exc:
            await update.message.reply_text(f"ثبت کاتالوگ ناموفق بود: {exc}", reply_markup=CATALOG_MANAGE_MENU)
            return CATALOG_ADMIN_ADD_CODE
        _reset_catalog_admin_context(context)
        await update.message.reply_text(
            f"کاتالوگ «{escape(title)}» با کد <code>{escape(code)}</code> ثبت شد. تعداد فایل: {len(files_list)}",
            reply_markup=MAIN_MENU_ADMIN,
            parse_mode="HTML",
        )
        return ConversationHandler.END
    if update.message.document:
        doc = update.message.document
        files_list.append((doc.file_id, doc.file_name))
        await update.message.reply_text("فایل دریافت شد. در صورت نیاز فایل دیگری بفرستید یا پایان بارگذاری را بزنید.", reply_markup=CATALOG_FILES_MENU)
        return CATALOG_ADMIN_ADD_FILES
    if update.message.photo:
        fid = update.message.photo[-1].file_id
        files_list.append((fid, None))
        await update.message.reply_text("تصویر دریافت شد. در صورت نیاز فایل‌های بیشتری بفرستید یا پایان بارگذاری را بزنید.", reply_markup=CATALOG_FILES_MENU)
        return CATALOG_ADMIN_ADD_FILES
    await update.message.reply_text("لطفاً فایل ارسال کنید یا گزینه پایان بارگذاری را انتخاب کنید.", reply_markup=CATALOG_FILES_MENU)
    return CATALOG_ADMIN_ADD_FILES


@require_membership(is_admin, admin_only=True)
async def catalog_admin_edit_select(update: Update, context: ContextTypes.DEFAULT_TYPE):
    raw_code = update.message.text or ""
    code = _normalize_catalog_code(raw_code)
    catalog = get_catalog_by_code(code)
    if not catalog:
        await update.message.reply_text("کاتالوگی با این کد یافت نشد.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_EDIT_SELECT
    context.user_data["catalog_edit_target"] = catalog[0]
    context.user_data["catalog_edit_original_title"] = catalog[1]
    context.user_data["catalog_edit_files"] = list_catalog_files(code)
    await update.message.reply_text(
        f"عنوان فعلی: «{escape(catalog[1])}». عنوان جدید را وارد کنید (برای حفظ عنوان فعلی همان را ارسال کنید).",
        reply_markup=CATALOG_MANAGE_MENU,
        parse_mode="HTML",
    )
    return CATALOG_ADMIN_EDIT_TITLE


@require_membership(is_admin, admin_only=True)
async def catalog_admin_edit_title(update: Update, context: ContextTypes.DEFAULT_TYPE):
    title = (update.message.text or "").strip()
    if not title:
        title = context.user_data.get("catalog_edit_original_title", "")
    context.user_data["catalog_edit_new_title"] = title
    await update.message.reply_text(
        "کد جدید کالا را وارد کنید (برای عدم تغییر همان کد فعلی را ارسال کنید).",
        reply_markup=CATALOG_MANAGE_MENU,
    )
    return CATALOG_ADMIN_EDIT_CODE


@require_membership(is_admin, admin_only=True)
async def catalog_admin_edit_code(update: Update, context: ContextTypes.DEFAULT_TYPE):
    target_code = context.user_data.get("catalog_edit_target")
    if not target_code:
        return await cancel(update, context)
    raw_code = update.message.text or ""
    new_code = _normalize_catalog_code(raw_code)
    if not new_code:
        await update.message.reply_text("کد کالا معتبر نیست.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_EDIT_CODE
    existing = get_catalog_by_code(new_code)
    if existing and existing[0] != target_code:
        await update.message.reply_text("این کد قبلاً برای کاتالوگ دیگری ثبت شده است.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_EDIT_CODE
    context.user_data["catalog_edit_new_code"] = new_code
    context.user_data["catalog_edit_new_files"] = []
    await update.message.reply_text(
        "فایل‌های جدید را برای جایگزینی ارسال کنید و در پایان «پایان بارگذاری» را بزنید.\n"
        "اگر نمی‌خواهید فایل‌ها تغییر کنند، «حفظ فایل های فعلی» را انتخاب کنید.",
        reply_markup=CATALOG_FILES_EDIT_MENU,
    )
    return CATALOG_ADMIN_EDIT_FILES


@require_membership(is_admin, admin_only=True)
async def catalog_admin_edit_files(update: Update, context: ContextTypes.DEFAULT_TYPE):
    text = _norm((update.message.text or "").strip())
    if is_back_text(update.message.text):
        _reset_catalog_admin_context(context)
        return await cancel(update, context)
    target_code = context.user_data.get("catalog_edit_target")
    if not target_code:
        return await cancel(update, context)
    new_files = context.user_data.setdefault("catalog_edit_new_files", [])
    if text == _norm(LBL_KEEP_FILES):
        files_to_use = context.user_data.get("catalog_edit_files", [])
    elif text == _norm(LBL_UPLOAD_DONE):
        if not new_files:
            await update.message.reply_text(
                "هیچ فایلی برای جایگزینی ثبت نشده است. فایل بفرستید یا گزینه حفظ فایل های فعلی را انتخاب کنید.",
                reply_markup=CATALOG_FILES_EDIT_MENU,
            )
            return CATALOG_ADMIN_EDIT_FILES
        files_to_use = new_files
    elif update.message.document:
        doc = update.message.document
        new_files.append((doc.file_id, doc.file_name))
        await update.message.reply_text("فایل دریافت شد. می‌توانید فایل دیگری بفرستید یا پایان بارگذاری را بزنید.", reply_markup=CATALOG_FILES_EDIT_MENU)
        return CATALOG_ADMIN_EDIT_FILES
    elif update.message.photo:
        fid = update.message.photo[-1].file_id
        new_files.append((fid, None))
        await update.message.reply_text("تصویر دریافت شد. می‌توانید فایل دیگری بفرستید یا پایان بارگذاری را بزنید.", reply_markup=CATALOG_FILES_EDIT_MENU)
        return CATALOG_ADMIN_EDIT_FILES
    else:
        await update.message.reply_text("لطفاً فایل ارسال کنید یا یکی از گزینه‌های کیبورد را انتخاب کنید.", reply_markup=CATALOG_FILES_EDIT_MENU)
        return CATALOG_ADMIN_EDIT_FILES

    title = context.user_data.get("catalog_edit_new_title") or context.user_data.get("catalog_edit_original_title", "")
    new_code = context.user_data.get("catalog_edit_new_code") or target_code
    try:
        updated = update_catalog(target_code, new_title=title, new_code=new_code)
    except Exception:
        await update.message.reply_text("این کد قبلاً ثبت شده است.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_EDIT_CODE
    if not updated:
        await update.message.reply_text("تغییری ذخیره نشد.", reply_markup=MAIN_MENU_ADMIN)
        _reset_catalog_admin_context(context)
        return ConversationHandler.END
    replace_catalog_files(new_code, files_to_use)
    _reset_catalog_admin_context(context)
    await update.message.reply_text(
        f"کاتالوگ به‌روزرسانی شد: «{escape(title)}» — <code>{escape(new_code)}</code>",
        reply_markup=MAIN_MENU_ADMIN,
        parse_mode="HTML",
    )
    return ConversationHandler.END


@require_membership(is_admin, admin_only=True)
async def catalog_admin_delete(update: Update, context: ContextTypes.DEFAULT_TYPE):
    if is_back_text(update.message.text):
        _reset_catalog_admin_context(context)
        return await cancel(update, context)
    raw_code = update.message.text or ""
    code = _normalize_catalog_code(raw_code)
    if not code:
        await update.message.reply_text("کد کالا معتبر نیست.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_DELETE_CONFIRM
    catalog = get_catalog_by_code(code)
    if not catalog:
        await update.message.reply_text("کاتالوگی با این کد یافت نشد.", reply_markup=CATALOG_MANAGE_MENU)
        return CATALOG_ADMIN_DELETE_CONFIRM
    file_count = len(list_catalog_files(code))
    deleted = delete_catalog(code)
    _reset_catalog_admin_context(context)
    if deleted:
        await update.message.reply_text(
            f"کاتالوگ «{escape(catalog[1])}» با کد <code>{escape(code)}</code> حذف شد. ({file_count} فایل پاک شد)",
            reply_markup=MAIN_MENU_ADMIN,
            parse_mode="HTML",
        )
    else:
        await update.message.reply_text("حذفی انجام نشد.", reply_markup=MAIN_MENU_ADMIN)
    return ConversationHandler.END


@require_membership(is_admin)
async def my_guarantees(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """list current user's guarantees"""
    uid = update.message.from_user.id
    rows = get_by_user(uid)
    if not rows:
        await update.message.reply_text("برای شما گارانتی ثبت نشده است.")
        return
    for row in rows[:10]:
        j_ex = jdatetime.datetime.fromgregorian(datetime=datetime.fromisoformat(row[7])).strftime("%Y/%m/%d")
        store_label = _format_store_label(row[4])
        await update.message.reply_text(f"{row[0]} | {row[2]} | {row[3]} | {store_label} | {j_ex}")

@require_membership(is_admin, admin_only=True)
async def list_guarantees(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """list recent guarantees (paginated by chunk size)"""
    limit = 20
    rows = list_all(limit)
    if not rows:
        await update.message.reply_text("موردی یافت نشد.")
        return
    lines: List[str] = []
    for row in rows:
        j_ex = jdatetime.datetime.fromgregorian(datetime=datetime.fromisoformat(row[7])).strftime("%Y/%m/%d")
        store_label = _format_store_label(row[4])
        lines.append(f"{row[0]} | {row[2]} | {row[3]} | {store_label} | {j_ex}")
    chunk: List[str] = []
    for line in lines:
        chunk.append(line)
        if len("\n".join(chunk)) > 3500 or len(chunk) >= 30:
            await update.message.reply_text("\n".join(chunk))
            chunk = []
    if chunk:
        await update.message.reply_text("\n".join(chunk))

@require_membership(is_admin, admin_only=True)
async def export_excel(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """export all to xlsx and send (in-memory; no disk)"""
    df = export_all()
    # write to BytesIO using openpyxl engine
    buf = BytesIO()
    with pd.ExcelWriter(buf, engine="openpyxl") as writer:
        df.to_excel(writer, index=False, sheet_name="guarantees")
    buf.seek(0)
    # send as document without saving on disk
    await update.message.reply_document(
        document=InputFile(buf, filename="guarantees_export.xlsx"),
        caption="خروجی اکسل آماده شد."
    )

# ------------- main -------------
def main() -> None:
    """bootstrap app and register handlers"""
    init_db()
    request = HTTPXRequest(connect_timeout=20, read_timeout=40, write_timeout=20, pool_timeout=5)
    app = ApplicationBuilder().token(TOKEN).request(request).build()

    # registration flow
    reg_conv = ConversationHandler(
        entry_points=[
            CommandHandler("new", start_new),
            MessageHandler(filters.Regex(exact_label_pattern(LBL_NEW)), start_new),
        ],
        states={
            PHOTO:  [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                     MessageHandler(filters.PHOTO, receive_photo)],
            INVOICE_DATE: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                           MessageHandler(filters.TEXT & ~filters.COMMAND, receive_invoice_date)],
            STORE_CODE:  [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                     MessageHandler(filters.TEXT & ~filters.COMMAND, receive_store)],
            NAME:   [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                     MessageHandler(filters.TEXT & ~filters.COMMAND, receive_name)],
            MOBILE: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                     MessageHandler(filters.TEXT & ~filters.COMMAND, receive_mobile)],
        },
        fallbacks=[MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                   CommandHandler("menu", cancel),
                   CommandHandler("start", cancel)],
    )

    # search flow
    search_conv = ConversationHandler(
        entry_points=[MessageHandler(filters.Regex(exact_label_pattern(LBL_SEARCH)), search_start)],
        states={
            SEARCH_GUARANTEE: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                     MessageHandler(filters.TEXT & ~filters.COMMAND, search_receive)]
        },
        fallbacks=[MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                   CommandHandler("menu", cancel),
                   CommandHandler("start", cancel)],
    )

    store_search_conv = ConversationHandler(
        entry_points=[MessageHandler(filters.Regex(exact_label_pattern(LBL_STORE_SEARCH)), store_search_start)],
        states={
            STORE_SEARCH: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                           MessageHandler(filters.TEXT & ~filters.COMMAND, store_search_receive)]
        },
        fallbacks=[MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                   CommandHandler("menu", cancel),
                   CommandHandler("start", cancel)],
    )

    catalog_search_conv = ConversationHandler(
        entry_points=[MessageHandler(filters.Regex(exact_label_pattern(LBL_CATALOG_SEARCH)), catalog_search_start)],
        states={
            CATALOG_SEARCH: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                             MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_search_receive)]
        },
        fallbacks=[MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                   CommandHandler("menu", cancel),
                   CommandHandler("start", cancel)],
    )

    store_manage_conv = ConversationHandler(
        entry_points=[MessageHandler(filters.Regex(exact_label_pattern(LBL_MANAGE_STORES)), store_manage_start)],
        states={
            STORE_ADMIN_MENU: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                               MessageHandler(filters.TEXT & ~filters.COMMAND, store_manage_action)],
            STORE_ADMIN_ADD_NAME: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                   MessageHandler(filters.TEXT & ~filters.COMMAND, store_admin_add_name)],
            STORE_ADMIN_ADD_CODE: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                   MessageHandler(filters.TEXT & ~filters.COMMAND, store_admin_add_code)],
            STORE_ADMIN_EDIT_SELECT: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                      MessageHandler(filters.TEXT & ~filters.COMMAND, store_admin_edit_select)],
            STORE_ADMIN_EDIT_NAME: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                   MessageHandler(filters.TEXT & ~filters.COMMAND, store_admin_edit_name)],
            STORE_ADMIN_EDIT_CODE: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                    MessageHandler(filters.TEXT & ~filters.COMMAND, store_admin_edit_code)],
        },
        fallbacks=[MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                   CommandHandler("menu", cancel),
                   CommandHandler("start", cancel)],
    )

    catalog_manage_conv = ConversationHandler(
        entry_points=[MessageHandler(filters.Regex(exact_label_pattern(LBL_MANAGE_CATALOGS)), catalog_manage_start)],
        states={
            CATALOG_ADMIN_MENU: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                 MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_manage_action)],
            CATALOG_ADMIN_ADD_TITLE: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                      MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_admin_add_title)],
            CATALOG_ADMIN_ADD_CODE: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                     MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_admin_add_code)],
            CATALOG_ADMIN_ADD_FILES: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                      MessageHandler(filters.Document.ALL, catalog_admin_add_files),
                                      MessageHandler(filters.PHOTO, catalog_admin_add_files),
                                      MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_admin_add_files)],
            CATALOG_ADMIN_EDIT_SELECT: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                        MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_admin_edit_select)],
            CATALOG_ADMIN_EDIT_TITLE: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                       MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_admin_edit_title)],
            CATALOG_ADMIN_EDIT_CODE: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                      MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_admin_edit_code)],
            CATALOG_ADMIN_EDIT_FILES: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                       MessageHandler(filters.Document.ALL, catalog_admin_edit_files),
                                       MessageHandler(filters.PHOTO, catalog_admin_edit_files),
                                       MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_admin_edit_files)],
            CATALOG_ADMIN_DELETE_CONFIRM: [MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                                           MessageHandler(filters.TEXT & ~filters.COMMAND, catalog_admin_delete)],
        },
        fallbacks=[MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), cancel),
                   CommandHandler("menu", cancel),
                   CommandHandler("start", cancel)],
    )

    app.add_handler(CallbackQueryHandler(catalog_send_files_callback, pattern=r"^catalog:"))

    if CHANNEL_MEMBERSHIP_REQUIRED:
        # 'I've joined' button (membership re-check)
        app.add_handler(CallbackQueryHandler(lambda u, c: on_recheck_join(u, c, is_admin), pattern=r"^recheck_join$"))

    # menus & commands
    app.add_handler(CommandHandler("start", start))
    app.add_handler(CommandHandler("menu", start))

    app.add_handler(reg_conv)
    app.add_handler(search_conv)
    app.add_handler(store_search_conv)
    app.add_handler(catalog_search_conv)
    app.add_handler(store_manage_conv)
    app.add_handler(catalog_manage_conv)

    app.add_handler(MessageHandler(filters.Regex(exact_label_pattern(LBL_BACK)), start))
    app.add_handler(MessageHandler(filters.Regex(exact_label_pattern(LBL_MINE)),         my_guarantees))
    app.add_handler(MessageHandler(filters.Regex(exact_label_pattern(LBL_LIST_RECENTS)), list_guarantees))
    app.add_handler(MessageHandler(filters.Regex(exact_label_pattern(LBL_EXPORT_XLSX)),  export_excel))
    app.add_handler(MessageHandler(filters.Regex(exact_label_pattern(LBL_HELP)),         start))

    app.add_handler(CommandHandler("check", check))
    app.add_handler(CommandHandler("search", search_receive))
    app.add_handler(CommandHandler("my", my_guarantees))
    app.add_handler(CommandHandler("list", list_guarantees))
    app.add_handler(CommandHandler("export", export_excel))

    # global error handler (notify admins)
    async def error_handler(update: object, context: ContextTypes.DEFAULT_TYPE) -> None:
        logging.exception("Unhandled exception", exc_info=context.error)
        for admin_id in ADMIN_IDS:
            try:
                await context.bot.send_message(chat_id=admin_id, text=f"Error: {context.error}")
            except Exception:
                continue

    app.add_error_handler(error_handler)
    app.run_polling()

if __name__ == "__main__":
    main()
