from __future__ import annotations
import uuid
from datetime import datetime
from typing import List, Optional
from sqlalchemy import (
    Boolean,
    BigInteger,
    CheckConstraint,
    DateTime,
    Enum,
    ForeignKey,
    Integer,
    JSON,
    Numeric,
    String,
    Text,
    UniqueConstraint,
)
from sqlalchemy.orm import Mapped, mapped_column, relationship
from app.db.base import Base
from app.db.types import GUID


def utcnow() -> datetime:
    return datetime.utcnow()


class TimestampMixin:
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=utcnow, nullable=False)
    updated_at: Mapped[datetime] = mapped_column(
        DateTime(timezone=True), default=utcnow, onupdate=utcnow, nullable=False
    )


class Tenant(Base, TimestampMixin):
    __tablename__ = "tenant"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    code: Mapped[str] = mapped_column(String(255), unique=True, nullable=False)
    title: Mapped[str] = mapped_column(Text, nullable=False)
    category: Mapped[Optional[str]]
    address: Mapped[Optional[str]]
    geo_location: Mapped[Optional[dict]] = mapped_column(JSON)
    work_hours: Mapped[Optional[str]]
    subscription_tier: Mapped[str] = mapped_column(
        Enum("base", "silver", "gold", name="subscription_tier"), default="base", nullable=False
    )
    status: Mapped[str] = mapped_column(Enum("active", "inactive", name="tenant_status"), default="active")

    bots: Mapped[List["Bot"]] = relationship(back_populates="tenant", cascade="all, delete-orphan")
    users: Mapped[List["AppUser"]] = relationship(back_populates="tenant", cascade="all, delete-orphan")
    store_profile: Mapped[Optional["StoreProfile"]] = relationship(back_populates="tenant", uselist=False)
    categories: Mapped[List["Category"]] = relationship(back_populates="tenant", cascade="all, delete-orphan")
    products: Mapped[List["Product"]] = relationship(back_populates="tenant", cascade="all, delete-orphan")
    orders: Mapped[List["Order"]] = relationship(back_populates="tenant", cascade="all, delete-orphan")
    store_customers: Mapped[List["StoreCustomer"]] = relationship(back_populates="tenant", cascade="all, delete-orphan")
    store_visits: Mapped[List["StoreVisit"]] = relationship(back_populates="tenant", cascade="all, delete-orphan")
    product_likes: Mapped[List["ProductLike"]] = relationship(back_populates="tenant", cascade="all, delete-orphan")


class Bot(Base, TimestampMixin):
    __tablename__ = "bot"
    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[Optional[uuid.UUID]] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"))
    type: Mapped[str] = mapped_column(Enum("hyper", "store", name="bot_type"), nullable=False)
    token: Mapped[str] = mapped_column(String(255), unique=True, nullable=False)
    username: Mapped[Optional[str]]
    verified_at: Mapped[Optional[datetime]]
    is_active: Mapped[bool] = mapped_column(Boolean, default=True)

    tenant: Mapped[Optional[Tenant]] = relationship(back_populates="bots")


class AppUser(Base, TimestampMixin):
    __tablename__ = "app_user"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[Optional[uuid.UUID]] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"))
    telegram_user_id: Mapped[Optional[int]]
    login_username: Mapped[Optional[str]] = mapped_column(String(255))
    password_hash: Mapped[Optional[str]] = mapped_column(Text)
    phone: Mapped[Optional[str]] = mapped_column(Text)
    role: Mapped[str] = mapped_column(
        Enum("hyper_admin", "store_admin", "store_staff", "customer", name="user_role"), nullable=False
    )
    status: Mapped[str] = mapped_column(Enum("active", "disabled", name="user_status"), default="active")

    tenant: Mapped[Optional[Tenant]] = relationship(back_populates="users")
    orders: Mapped[List["Order"]] = relationship(back_populates="customer")

    __table_args__ = (UniqueConstraint("tenant_id", "login_username", name="uq_user_tenant_username"),)


class StoreProfile(Base):
    __tablename__ = "store_profile"
    tenant_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"), primary_key=True)
    about_text: Mapped[Optional[str]]
    support_text: Mapped[Optional[str]]
    contact_phone: Mapped[Optional[str]]
    logo_file_id: Mapped[Optional[str]]
    channel_id: Mapped[Optional[str]]
    channel_invite_link: Mapped[Optional[str]]
    manager_telegram_ids: Mapped[Optional[list[int]]] = mapped_column(JSON, default=list)
    payment_link_template: Mapped[Optional[str]]
    payment_link_note: Mapped[Optional[str]]
    updated_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=utcnow, onupdate=utcnow, nullable=False)

    tenant: Mapped[Tenant] = relationship(back_populates="store_profile")


class StoreVisit(Base, TimestampMixin):
    __tablename__ = "store_visit"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False)
    telegram_user_id: Mapped[int] = mapped_column(BigInteger, nullable=False)
    session_token: Mapped[str] = mapped_column(Text, unique=True, nullable=False, default=lambda: str(uuid.uuid4()))
    started_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=utcnow, nullable=False)
    ended_at: Mapped[Optional[datetime]] = mapped_column(DateTime(timezone=True))
    status: Mapped[str] = mapped_column(
        Enum("active", "completed", "cancelled", name="store_visit_status"), default="active", nullable=False
    )

    tenant: Mapped[Tenant] = relationship(back_populates="store_visits")
    likes: Mapped[List["ProductLike"]] = relationship(back_populates="session", cascade="all, delete-orphan")


class ProductLike(Base, TimestampMixin):
    __tablename__ = "product_like"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False)
    product_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("product.id", ondelete="CASCADE"), nullable=False)
    telegram_user_id: Mapped[int] = mapped_column(BigInteger, nullable=False)
    session_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("store_visit.id", ondelete="CASCADE"))
    liked_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=utcnow, nullable=False)

    tenant: Mapped[Tenant] = relationship(back_populates="product_likes")
    product: Mapped["Product"] = relationship(back_populates="likes")
    session: Mapped[Optional[StoreVisit]] = relationship(back_populates="likes")

    __table_args__ = (
        UniqueConstraint("tenant_id", "product_id", "telegram_user_id", name="uq_product_like_user"),
    )


class AccountRequest(Base):
    __tablename__ = "account_request"
    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    full_name: Mapped[Optional[str]]
    phone: Mapped[Optional[str]]
    description: Mapped[Optional[str]]
    status: Mapped[str] = mapped_column(
        Enum("new", "in_progress", "done", "rejected", name="account_request_status"), default="new"
    )
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=utcnow, nullable=False)


class Category(Base, TimestampMixin):
    __tablename__ = "category"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False)
    name: Mapped[str] = mapped_column(String(255), nullable=False)
    parent_id: Mapped[Optional[uuid.UUID]] = mapped_column(GUID(), ForeignKey("category.id", ondelete="SET NULL"))

    tenant: Mapped[Tenant] = relationship(back_populates="categories")
    parent: Mapped[Optional["Category"]] = relationship(remote_side="Category.id")
    products: Mapped[List["Product"]] = relationship(secondary="product_category", back_populates="categories")

    __table_args__ = (UniqueConstraint("tenant_id", "name", name="uq_category_tenant_name"),)


class Product(Base, TimestampMixin):
    __tablename__ = "product"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False)
    sku: Mapped[str] = mapped_column(String(255), nullable=False)
    title: Mapped[str] = mapped_column(Text, nullable=False)
    name: Mapped[Optional[str]]
    description: Mapped[Optional[str]]
    base_price: Mapped[float] = mapped_column(Numeric(12, 2), nullable=False)
    image_file_id: Mapped[Optional[str]]
    active: Mapped[bool] = mapped_column(Boolean, default=True)
    stock: Mapped[int] = mapped_column(Integer, default=0)

    tenant: Mapped[Tenant] = relationship(back_populates="products")
    images: Mapped[List["ProductImage"]] = relationship(back_populates="product", cascade="all, delete-orphan")
    specs: Mapped[List["ProductSpec"]] = relationship(back_populates="product", cascade="all, delete-orphan")
    price_history: Mapped[List["ProductPriceHistory"]] = relationship(back_populates="product", cascade="all, delete-orphan")
    categories: Mapped[List[Category]] = relationship(secondary="product_category", back_populates="products")
    likes: Mapped[List["ProductLike"]] = relationship(back_populates="product", cascade="all, delete-orphan")

    __table_args__ = (UniqueConstraint("tenant_id", "sku", name="uq_product_tenant_sku"),)


class ProductImage(Base):
    __tablename__ = "product_image"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    product_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("product.id", ondelete="CASCADE"), nullable=False)
    telegram_file_id: Mapped[str] = mapped_column(Text, nullable=False)
    sort_order: Mapped[int] = mapped_column(Integer, default=0)

    product: Mapped[Product] = relationship(back_populates="images")


class ProductSpec(Base):
    __tablename__ = "product_spec"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    product_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("product.id", ondelete="CASCADE"), nullable=False)
    name: Mapped[str] = mapped_column(String(255), nullable=False)
    values: Mapped[str] = mapped_column(Text, nullable=False)

    product: Mapped[Product] = relationship(back_populates="specs")

    __table_args__ = (UniqueConstraint("product_id", "name", name="uq_product_spec_name"),)


class ProductCategory(Base):
    __tablename__ = "product_category"
    product_id: Mapped[uuid.UUID] = mapped_column(
        GUID(), ForeignKey("product.id", ondelete="CASCADE"), primary_key=True
    )
    category_id: Mapped[uuid.UUID] = mapped_column(
        GUID(), ForeignKey("category.id", ondelete="CASCADE"), primary_key=True
    )


class ProductPriceHistory(Base):
    __tablename__ = "product_price_history"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    product_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("product.id", ondelete="CASCADE"), nullable=False)
    base_price: Mapped[float] = mapped_column(Numeric(12, 2), nullable=False)
    discount_percent: Mapped[int] = mapped_column(Integer, default=0)
    effective_from: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=utcnow, nullable=False)
    effective_to: Mapped[Optional[datetime]] = mapped_column(DateTime(timezone=True))

    product: Mapped[Product] = relationship(back_populates="price_history")

    __table_args__ = (
        CheckConstraint("discount_percent BETWEEN 0 AND 90", name="ck_discount_percent_range"),
        CheckConstraint("base_price >= 0", name="ck_pph_price_positive"),
    )


class StoreCustomer(Base, TimestampMixin):
    __tablename__ = "store_customer"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False)
    user_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("app_user.id", ondelete="CASCADE"), nullable=False)
    first_name: Mapped[str] = mapped_column(Text, nullable=False)
    last_name: Mapped[str] = mapped_column(Text, nullable=False)
    phone: Mapped[str] = mapped_column(Text, nullable=False)
    postal_code: Mapped[Optional[str]]
    address: Mapped[Optional[str]]
    delivery_geo: Mapped[Optional[dict]] = mapped_column(JSON)

    tenant: Mapped[Tenant] = relationship(back_populates="store_customers")
    user: Mapped[AppUser] = relationship()

    __table_args__ = (UniqueConstraint("tenant_id", "user_id", name="uq_store_customer_tenant_user"),)


class Order(Base, TimestampMixin):
    __tablename__ = "order"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False)
    customer_user_id: Mapped[Optional[uuid.UUID]] = mapped_column(GUID(), ForeignKey("app_user.id", ondelete="SET NULL"))
    status: Mapped[str] = mapped_column(
        Enum("cart", "saved", "awaiting_payment", "paid", "canceled", name="order_status"), default="cart"
    )
    payment_method: Mapped[Optional[str]] = mapped_column(
        Enum("card2card", "online", "inperson", name="order_payment_method")
    )
    total: Mapped[float] = mapped_column(Numeric(12, 2), default=0)

    tenant: Mapped[Tenant] = relationship(back_populates="orders")
    customer: Mapped[Optional[AppUser]] = relationship(back_populates="orders")
    items: Mapped[List["OrderItem"]] = relationship(back_populates="order", cascade="all, delete-orphan")
    payments: Mapped[List["Payment"]] = relationship(back_populates="order", cascade="all, delete-orphan")


class OrderItem(Base):
    __tablename__ = "order_item"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False)
    order_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("order.id", ondelete="CASCADE"), nullable=False)
    product_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("product.id"), nullable=False)
    qty: Mapped[int] = mapped_column(Integer, nullable=False, default=1)
    unit_price_snapshot: Mapped[float] = mapped_column(Numeric(12, 2), nullable=False)
    discount_percent_snapshot: Mapped[int] = mapped_column(Integer, default=0)
    final_price_snapshot: Mapped[float] = mapped_column(Numeric(12, 2), nullable=False)
    chosen_specs_json: Mapped[Optional[dict]] = mapped_column(JSON)
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=utcnow, nullable=False)

    order: Mapped[Order] = relationship(back_populates="items")
    product: Mapped[Product] = relationship()

    __table_args__ = (
        CheckConstraint("qty > 0", name="ck_order_item_qty_positive"),
        CheckConstraint("discount_percent_snapshot BETWEEN 0 AND 90", name="ck_order_item_discount_range"),
    )


class Payment(Base):
    __tablename__ = "payment"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False)
    order_id: Mapped[uuid.UUID] = mapped_column(GUID(), ForeignKey("order.id", ondelete="CASCADE"), nullable=False)
    method: Mapped[str] = mapped_column(Enum("card2card", "online", "inperson", name="payment_method"), nullable=False)
    amount: Mapped[float] = mapped_column(Numeric(12, 2), nullable=False)
    gateway_tx_id: Mapped[Optional[str]]
    card2card_note: Mapped[Optional[str]]
    status: Mapped[str] = mapped_column(
        Enum("pending", "confirmed", "failed", name="payment_status"), default="pending", nullable=False
    )
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=utcnow, nullable=False)

    order: Mapped[Order] = relationship(back_populates="payments")

    __table_args__ = (CheckConstraint("amount >= 0", name="ck_payment_amount_positive"),)


class AuditLog(Base):
    __tablename__ = "audit_log"

    id: Mapped[uuid.UUID] = mapped_column(GUID(), primary_key=True, default=uuid.uuid4)
    tenant_id: Mapped[Optional[uuid.UUID]] = mapped_column(GUID(), ForeignKey("tenant.id", ondelete="CASCADE"))
    actor_user_id: Mapped[Optional[uuid.UUID]] = mapped_column(GUID(), ForeignKey("app_user.id", ondelete="SET NULL"))
    action: Mapped[str] = mapped_column(Text, nullable=False)
    payload: Mapped[Optional[dict]] = mapped_column(JSON)
    created_at: Mapped[datetime] = mapped_column(DateTime(timezone=True), default=utcnow, nullable=False)


__all__ = [
    "Tenant",
    "Bot",
    "AppUser",
    "StoreProfile",
    "AccountRequest",
    "Category",
    "Product",
    "ProductImage",
    "ProductSpec",
    "ProductCategory",
    "ProductPriceHistory",
    "StoreCustomer",
    "Order",
    "OrderItem",
    "Payment",
    "AuditLog",
]
