from __future__ import annotations

from typing import Any

from aiogram import Router, F, types
from aiogram.filters import CommandStart, Command
from aiogram.fsm.context import FSMContext

from app.bots.hyper.states import (
    AdminStates,
    RegisterMarketStates,
    ManageTenantStates,
    BaseSettingsStates,
)
from app.bots.hyper import keyboards
from app.bots.shared.api_client import BackendClient
from app.core.config import get_settings
from app.core.logging import get_logger

router = Router()
logger = get_logger("app.bots.hyper")
settings = get_settings()


# ---------------------------------------------------------------------------
# پیکربندی ادمین‌ها
# ---------------------------------------------------------------------------


def _parse_allowed_admins(raw: str | None) -> set[int]:
    if not raw:
        return set()
    result: set[int] = set()
    for chunk in raw.split(","):
        chunk = chunk.strip()
        if not chunk:
            continue
        try:
            result.add(int(chunk))
        except ValueError:
            continue
    return result


ALLOWED_ADMIN_IDS = _parse_allowed_admins(settings.HYPER_ADMIN_TELEGRAM_IDS)


# ---------------------------------------------------------------------------
# تعریف جریان ثبت مارکت جدید
# ---------------------------------------------------------------------------

REGISTER_FLOW: list[dict[str, Any]] = [
    {"key": "title", "label": "نام مارکت", "type": "text"},
    {"key": "store_admin_phone", "label": "موبایل مدیر مارکت", "type": "text"},
    {"key": "address", "label": "آدرس مارکت", "type": "text"},
    {"key": "category", "label": "دسته‌بندی صنفی", "type": "text"},
    {"key": "bot_token", "label": "توکن ربات فروشگاه", "type": "text"},
    {"key": "channel_id", "label": "شناسه کانال فروشگاه", "type": "text"},
    {"key": "channel_invite_link", "label": "لینک دعوت کانال", "type": "text"},
    {"key": "work_hours", "label": "ساعات کاری", "type": "text"},
    {"key": "subscription_tier", "label": "پلن اشتراک", "type": "choice"},
]

REGISTER_MAP: dict[str, dict[str, Any]] = {
    step["key"]: step for step in REGISTER_FLOW
}


# فیلدهای قابل ویرایش در پروفایل مارکت
EDITABLE_TENANT_FIELDS: list[dict[str, Any]] = [
    {"key": "title", "label": "عنوان مارکت", "scope": "tenant", "type": "text"},
    {"key": "category", "label": "دسته‌بندی", "scope": "tenant", "type": "text"},
    {"key": "address", "label": "آدرس", "scope": "tenant", "type": "text"},
    {"key": "work_hours", "label": "ساعات کاری", "scope": "tenant", "type": "text"},
    {
        "key": "subscription_tier",
        "label": "پلن اشتراک",
        "scope": "tenant",
        "type": "choice",
    },
    {
        "key": "contact_phone",
        "label": "تلفن تماس",
        "scope": "profile",
        "type": "text",
    },
    {
        "key": "about_text",
        "label": "درباره فروشگاه",
        "scope": "profile",
        "type": "text",
    },
    {
        "key": "support_text",
        "label": "پشتیبانی فروشگاه",
        "scope": "profile",
        "type": "text",
    },
    {
        "key": "channel_id",
        "label": "شناسه کانال",
        "scope": "profile",
        "type": "text",
    },
    {
        "key": "channel_invite_link",
        "label": "لینک دعوت کانال",
        "scope": "profile",
        "type": "text",
    },
    {
        "key": "payment_link_template",
        "label": "قالب لینک پرداخت",
        "scope": "profile",
        "type": "text",
    },
    {
        "key": "payment_link_note",
        "label": "یادداشت لینک پرداخت",
        "scope": "profile",
        "type": "text",
    },
    {
        "key": "manager_telegram_ids",
        "label": "آیدی مدیران",
        "scope": "profile",
        "type": "text",
        "parser": "manager_ids",
    },
]

EDITABLE_FIELD_MAP: dict[str, dict[str, Any]] = {
    field["key"]: field for field in EDITABLE_TENANT_FIELDS
}
FIELD_KEY_ALIAS_MAP: dict[str, str] = {
    field["key"]: format(idx, "x") for idx, field in enumerate(EDITABLE_TENANT_FIELDS)
}
ALIAS_TO_FIELD_KEY: dict[str, str] = {
    alias: key for key, alias in FIELD_KEY_ALIAS_MAP.items()
}


# ---------------------------------------------------------------------------
# Helpers
# ---------------------------------------------------------------------------


async def _get_client(state: FSMContext) -> BackendClient:
    data = await state.get_data()
    token = data.get("token")
    if not token:
        raise RuntimeError("missing token")
    return BackendClient(token=token)


async def _reset_register_flow(state: FSMContext) -> None:
    await state.update_data(register_index=0, register_data={}, register_waiting=None)


async def _current_step(state: FSMContext) -> dict[str, Any] | None:
    data = await state.get_data()
    idx = data.get("register_index", 0)
    if idx >= len(REGISTER_FLOW):
        return None
    return REGISTER_FLOW[idx]


def _short_exception(exc: Exception | str, limit: int = 160) -> str:
    text = str(exc).replace("\n", " ").strip()
    if len(text) > limit:
        return text[: limit - 1] + "…"
    return text


async def _show_register_prompt(
    target: types.Message | types.CallbackQuery,
    state: FSMContext,
) -> None:
    """
    نمایش پیام مرحله فعلی ثبت مارکت.
    اگر به انتهای مراحل رسیده باشیم، خلاصه را نشان می‌دهد.
    """
    step = await _current_step(state)
    if step is None:
        await _show_register_summary(target, state)
        return

    data = await state.get_data()
    register_data: dict[str, Any] = data.get("register_data", {})
    idx = data.get("register_index", 0)

    text = f"مرحله {idx + 1}/{len(REGISTER_FLOW)} – {step['label']}"
    current_value = register_data.get(step["key"])
    if current_value:
        text += f"\nمقدار فعلی: {current_value}"

    if isinstance(target, types.CallbackQuery):
        message = target.message
    else:
        message = target

    if step["type"] == "choice":
        # پلن اشتراک
        await state.update_data(register_waiting=None)
        await state.set_state(RegisterMarketStates.flow)
        await message.answer(text, reply_markup=keyboards.register_tier_keyboard())
    else:
        # فیلد متنی
        await state.update_data(register_waiting=step["key"])
        await state.set_state(RegisterMarketStates.awaiting_value)
        # اینجا بر اساس کد اول از register_input_keyboard استفاده می‌کنیم
        await message.answer(
            text,
            reply_markup=keyboards.register_input_keyboard(step["key"]),
        )
        await message.answer(
            f"مقدار «{step['label']}» را ارسال کنید.\n"
            "برای لغو /cancel را بزنید."
        )


async def _show_register_summary(
    target: types.Message | types.CallbackQuery,
    state: FSMContext,
) -> None:
    """
    خلاصه اطلاعات وارد شده برای مارکت جدید + کیبورد تایید نهایی.
    """
    data = await state.get_data()
    register_data: dict[str, Any] = data.get("register_data", {})
    lines = [
        f"{REGISTER_MAP[key]['label']}: {value}" for key, value in register_data.items()
    ]
    summary = "پیش‌نمایش مارکت:\n" + "\n".join(lines)

    if isinstance(target, types.CallbackQuery):
        await target.message.answer(summary, reply_markup=keyboards.register_confirm_keyboard())
    else:
        await target.answer(summary, reply_markup=keyboards.register_confirm_keyboard())


async def _cancel_register_flow(
    message: types.Message | types.CallbackQuery,
    state: FSMContext,
) -> None:
    """
    لغو کامل جریان ثبت مارکت و برگشت به منوی مدیر هایپر.
    """
    await _reset_register_flow(state)
    await state.set_state(AdminStates.menu)
    text = "عملیات ثبت مارکت لغو شد."

    if isinstance(message, types.CallbackQuery):
        await message.answer()
        await message.message.answer(text, reply_markup=keyboards.hyper_admin_menu())
    else:
        await message.answer(text, reply_markup=keyboards.hyper_admin_menu())


def _profile_fields_keyboard(tenant_id: str) -> InlineKeyboardMarkup:
    from aiogram.types import InlineKeyboardMarkup, InlineKeyboardButton

    rows: list[list[InlineKeyboardButton]] = []
    for field in EDITABLE_TENANT_FIELDS:
        alias = FIELD_KEY_ALIAS_MAP[field["key"]]
        rows.append(
            [
                InlineKeyboardButton(
                    text=field["label"],
                    callback_data=f"tact:profile:{tenant_id}:{alias}",
                )
            ]
        )
    # متن دکمه بازگشت را با کد اول یکسان کردیم: «⬅️ بازگشت»
    rows.append(
        [InlineKeyboardButton(text="⬅️ بازگشت", callback_data="tenant:list")]
    )
    return InlineKeyboardMarkup(inline_keyboard=rows)


def _parse_manager_ids(raw: str) -> list[int]:
    import re

    values: list[int] = []
    for chunk in re.split(r"[\s,،]+", raw.strip()):
        if not chunk:
            continue
        try:
            values.append(int(chunk))
        except ValueError:
            continue
    return values


def _normalize_manager_ids(raw: list[int | str] | None) -> list[int]:
    if not raw:
        return []
    normalized: list[int] = []
    for value in raw:
        try:
            num = int(value)
        except (TypeError, ValueError):
            continue
        if num > 0:
            normalized.append(num)
    return normalized


def _format_tenant_summary(tenant: dict[str, Any]) -> str:
    status_map = {"active": "فعال", "inactive": "غیرفعال"}
    return "\n".join(
        [
            f"مارکت: {tenant.get('title')}",
            f"کد: {tenant.get('code')}",
            f"وضعیت: {status_map.get(tenant.get('status'), 'نامشخص')}",
            f"اشتراک: {tenant.get('subscription_tier')}",
            f"دسته‌بندی: {tenant.get('category') or '-'}",
            f"آدرس: {tenant.get('address') or '-'}",
            f"تماس: {tenant.get('contact_phone') or '-'}",
            f"ساعات کاری: {tenant.get('work_hours') or '-'}",
        ]
    )


async def _show_tenant_actions(
    target: types.Message | types.CallbackQuery,
    state: FSMContext,
    tenant: dict[str, Any],
    note: str | None = None,
) -> None:
    """
    نمایش خلاصه مارکت + کیبورد اکشن‌ها.
    """
    parts: list[str] = []
    if note:
        parts.append(note)
    parts.append(_format_tenant_summary(tenant))
    text = "\n\n".join(parts)

    keyboard = keyboards.tenant_actions_keyboard(
        tenant["id"],
        tenant.get("status") == "active",
    )

    if isinstance(target, types.CallbackQuery):
        await target.answer()
        await target.message.answer(text, reply_markup=keyboard)
    else:
        await target.answer(text, reply_markup=keyboard)

    await state.set_state(ManageTenantStates.action)
    await state.update_data(
        selected_tenant_id=tenant["id"],
        editing_field=None,
        editing_scope=None,
        editing_label=None,
    )


async def _apply_profile_update(
    target: types.Message | types.CallbackQuery,
    state: FSMContext,
    tenant_id: str,
    field_key: str,
    value: Any,
) -> None:
    """
    اعمال تغییر در /tenants/{id} یا /tenants/{id}/profile
    بسته به scope فیلد.
    """
    data = await state.get_data()
    scope = data.get("editing_scope", "profile")
    label = data.get("editing_label", field_key)
    path = f"/tenants/{tenant_id}" if scope == "tenant" else f"/tenants/{tenant_id}/profile"
    client = await _get_client(state)

    try:
        await client.put(path, {field_key: value})
    except Exception as exc:
        details = _short_exception(exc)
        error_text = f"خطا در بروزرسانی {label}: {details}"
        if isinstance(target, types.CallbackQuery):
            await target.answer(error_text, show_alert=True)
        else:
            await target.answer(error_text)
        return

    refreshed = await client.get(f"/tenants/{tenant_id}")
    await _show_tenant_actions(target, state, refreshed, f"{label} بروزرسانی شد.")


# ---------------------------------------------------------------------------
# Entry points
# ---------------------------------------------------------------------------


@router.message(CommandStart())
async def start(message: types.Message, state: FSMContext):
    """
    شروع ربات هایپر:
    - پاک کردن state
    - نمایش منوی اصلی برای کاربر (با دکمه ورود مدیر هایپر)
    """
    await state.clear()
    await message.answer(
        "به ربات مدیریت هایپر خوش آمدید.\n"
        "برای ادامه، یکی از گزینه‌های زیر را انتخاب کنید.",
        reply_markup=keyboards.main_menu(),
    )


@router.message(F.text == "🔐 ورود مدیر هایپر")
async def hyper_login(message: types.Message, state: FSMContext):
    """
    ورود خودکار مدیر هایپر با یوزر/پسورد پیش‌فرض از تنظیمات.
    فقط به آی‌دی‌های مجاز اجازه ورود می‌دهد.
    """
    if ALLOWED_ADMIN_IDS and message.from_user.id not in ALLOWED_ADMIN_IDS:
        await message.answer("دسترسی شما برای مدیریت هایپر مجاز نیست.")
        return

    waiting = await message.answer("⏳ در حال ورود...")
    client = BackendClient()

    try:
        auth = await client.login(
            settings.DEFAULT_HYPER_ADMIN_USERNAME,
            settings.DEFAULT_HYPER_ADMIN_PASSWORD,
        )
    except Exception as exc:
        details = _short_exception(exc)
        await waiting.edit_text(f"ورود ناموفق بود: {details}")
        logger.exception("hyper admin login failed")
        return

    await state.update_data(token=auth["token"])
    await state.set_state(AdminStates.menu)
    await waiting.edit_text("ورود موفقیت‌آمیز بود.")
    await message.answer(
        "از منوی زیر گزینه‌ی مورد نظر خود را انتخاب کنید.",
        reply_markup=keyboards.hyper_admin_menu(),
    )


@router.message(AdminStates.menu, F.text == "↩️ خروج")
async def hyper_logout(message: types.Message, state: FSMContext):
    """
    خروج مدیر هایپر و بازگشت به منوی کاربر عادی.
    """
    await state.clear()
    await message.answer("از حساب خارج شدید.", reply_markup=keyboards.main_menu())


@router.message(AdminStates.menu, F.text == "⚙️ تنظیمات پایه")
async def base_settings_menu(message: types.Message, state: FSMContext):
    tenant_id = settings.STORE_TENANT_ID
    if not tenant_id:
        await message.answer("STORE_TENANT_ID در تنظیمات تعریف نشده است.")
        return
    await state.set_state(BaseSettingsStates.menu)
    await message.answer(
        "کدام بخش را می‌خواهید ویرایش کنید؟",
        reply_markup=keyboards.base_settings_keyboard(),
    )


@router.callback_query(BaseSettingsStates.menu, F.data.startswith("base:"))
async def base_settings_select(callback: types.CallbackQuery, state: FSMContext):
    key = callback.data.split(":")[1]
    field_map = {"about": "about_text", "guide": "support_text"}
    label_map = {"about": "متن درباره ما", "guide": "متن راهنما"}
    field_key = field_map.get(key)
    label = label_map.get(key)
    tenant_id = settings.STORE_TENANT_ID
    if not field_key or not tenant_id:
        await callback.answer("تنظیمات قابل ویرایش یافت نشد.", show_alert=True)
        return
    client = await _get_client(state)
    try:
        profile = await client.get(f"/tenants/{tenant_id}/profile", tenant_id=tenant_id)
    except Exception as exc:
        details = _short_exception(exc)
        await callback.answer(f"دریافت اطلاعات ممکن نبود: {details}", show_alert=True)
        return
    current_value = profile.get(field_key) if profile else ""
    await state.update_data(
        base_setting_key=field_key,
        base_setting_label=label,
        base_setting_tenant_id=tenant_id,
    )
    await state.set_state(BaseSettingsStates.awaiting_value)
    await callback.answer()
    await callback.message.answer(
        f"{label} فعلی:\n{current_value or 'ثبت نشده'}\n\nاگر می‌خواهید مقدار جدید ثبت شود، آن را ارسال کنید یا /cancel بنویسید."
    )


@router.callback_query(BaseSettingsStates.menu, F.data == "base:cancel")
async def base_settings_cancel(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(AdminStates.menu)
    await callback.answer()
    await callback.message.answer(
        "به منوی اصلی مدیر هایپر بازگشتید.",
        reply_markup=keyboards.hyper_admin_menu(),
    )


@router.message(BaseSettingsStates.awaiting_value)
async def base_settings_value(message: types.Message, state: FSMContext):
    data = await state.get_data()
    field_key = data.get("base_setting_key")
    label = data.get("base_setting_label") or "مقدار"
    tenant_id = data.get("base_setting_tenant_id") or settings.STORE_TENANT_ID
    if not (field_key and tenant_id):
        await message.answer("اطلاعات کافی برای بروزرسانی وجود ندارد.")
        return
    value = (message.text or "").strip()
    if not value:
        await message.answer("مقداری وارد نشده است.")
        return
    client = await _get_client(state)
    try:
        await client.put(f"/tenants/{tenant_id}/profile", {field_key: value}, tenant_id=tenant_id)
    except Exception as exc:
        details = _short_exception(exc)
        await message.answer(f"به‌روزرسانی انجام نشد: {details}")
        return
    await state.set_state(BaseSettingsStates.menu)
    await message.answer(
        f"{label} به‌روزرسانی شد.",
        reply_markup=keyboards.base_settings_keyboard(),
    )


@router.message(BaseSettingsStates.awaiting_value, Command("cancel"))
async def base_settings_cancel_input(message: types.Message, state: FSMContext):
    await state.set_state(BaseSettingsStates.menu)
    await message.answer(
        "لغو شد.",
        reply_markup=keyboards.base_settings_keyboard(),
    )


# ---------------------------------------------------------------------------
# Register market flow
# ---------------------------------------------------------------------------


@router.message(AdminStates.menu, F.text == "➕ ثبت مارکت جدید")
async def register_market_entry(message: types.Message, state: FSMContext):
    """
    شروع فرآیند ثبت مارکت جدید.
    """
    await _reset_register_flow(state)
    await state.set_state(RegisterMarketStates.flow)
    await message.answer(
        "وارد فرآیند ثبت مارکت جدید شدید.\n"
        "لطفاً اطلاعات را مرحله به مرحله ارسال کنید.",
    )
    await _show_register_prompt(message, state)


@router.message(RegisterMarketStates.awaiting_value)
async def register_receive_value(message: types.Message, state: FSMContext):
    """
    دریافت مقدار هر فیلد ثبت مارکت.
    """
    data = await state.get_data()
    step_key = data.get("register_waiting")
    if not step_key:
        await message.answer("مرحله‌ای برای دریافت مقدار فعال نیست.")
        return

    value = (message.text or "").strip()
    if not value:
        await message.answer("مقداری وارد نشده است.")
        return

    register_data = data.get("register_data", {})
    register_data[step_key] = value
    await state.update_data(register_data=register_data, register_waiting=None)

    # رفتن به مرحله بعد
    idx = data.get("register_index", 0)
    await state.update_data(register_index=idx + 1)
    await state.set_state(RegisterMarketStates.flow)
    await _show_register_prompt(message, state)


@router.message(RegisterMarketStates.awaiting_value, Command("cancel"))
async def register_cancel_command(message: types.Message, state: FSMContext):
    await _cancel_register_flow(message, state)


@router.callback_query(RegisterMarketStates.flow, F.data == "reg:back")
async def register_back(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    idx = max(data.get("register_index", 0) - 1, 0)
    await state.update_data(register_index=idx)
    await callback.answer()
    await callback.message.answer("به مرحله قبل برگشتید.")
    await _show_register_prompt(callback, state)


@router.callback_query(RegisterMarketStates.awaiting_value, F.data == "reg:back")
async def register_back_from_input(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(RegisterMarketStates.flow)
    await state.update_data(register_waiting=None)
    await callback.answer()
    await _show_register_prompt(callback, state)


@router.callback_query(F.data == "reg:cancel")
async def register_cancel(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer()
    await _cancel_register_flow(callback, state)


@router.callback_query(RegisterMarketStates.flow, F.data.startswith("reg:tier:"))
async def register_set_tier(callback: types.CallbackQuery, state: FSMContext):
    """
    انتخاب پلن اشتراک و پرش به پایان مراحل ثبت (بعد از انتخاب، خلاصه نمایش داده می‌شود).
    """
    tier = callback.data.split(":")[-1]
    data = await state.get_data()
    register_data = data.get("register_data", {})
    register_data["subscription_tier"] = tier
    await state.update_data(register_data=register_data, register_index=len(REGISTER_FLOW))
    await callback.answer(f"اشتراک {tier} انتخاب شد.")
    await _show_register_prompt(callback, state)


@router.callback_query(RegisterMarketStates.flow, F.data == "reg:confirm")
async def register_confirm(callback: types.CallbackQuery, state: FSMContext):
    """
    تایید نهایی ثبت مارکت و ایجاد tenant در بک‌اند.
    """
    await callback.answer()

    data = await state.get_data()
    register_data: dict[str, Any] = data.get("register_data", {}) or {}

    payload = {
        "title": register_data.get("title"),
        "store_admin_phone": register_data.get("store_admin_phone"),
        "address": register_data.get("address"),
        "category": register_data.get("category"),
        "bot_token": register_data.get("bot_token"),
        "work_hours": register_data.get("work_hours"),
        "subscription_tier": register_data.get("subscription_tier", "base"),
    }

    client = await _get_client(state)
    waiting = await callback.message.answer("⏳ در حال ثبت مارکت...")

    try:
        resp = await client.post("/tenants", payload)
        tenant_id = resp["id"]

        channel_payload: dict[str, Any] = {}
        if register_data.get("channel_id"):
            channel_payload["channel_id"] = register_data["channel_id"]
        if register_data.get("channel_invite_link"):
            channel_payload["channel_invite_link"] = register_data["channel_invite_link"]

        if channel_payload:
            await client.put(f"/tenants/{tenant_id}/profile", channel_payload)

    except Exception as exc:
        logger.exception("register_confirm failed", exc_info=exc)
        try:
            details = _short_exception(exc)
            await waiting.edit_text(f"ثبت مارکت ناموفق بود: {details}")
        except Exception:
            details = _short_exception(exc)
            await callback.message.answer(f"ثبت مارکت ناموفق بود: {details}")
        return

    success_text = (
        "✅ مارکت با موفقیت ایجاد شد.\n"
        f"کد مارکت: {resp['code']}\n"
        f"نام کاربری مدیر: {resp['admin_credentials']['username']}"
    )

    try:
        await waiting.edit_text(success_text)
    except Exception:
        await callback.message.answer(success_text)

    await callback.message.answer(
        "به منوی مدیریت هایپر برگشتید. از گزینه‌های زیر استفاده کنید:",
        reply_markup=keyboards.hyper_admin_menu(),
    )

    await state.set_state(AdminStates.menu)
    await _reset_register_flow(state)


# ---------------------------------------------------------------------------
# مدیریت مارکت‌ها (لیست و خلاصه و اکشن‌ها)
# ---------------------------------------------------------------------------


@router.message(AdminStates.menu, F.text == "📂 مدیریت مارکت‌ها")
async def tenants_manage(message: types.Message, state: FSMContext):
    client = await _get_client(state)
    tenants = await client.get("/tenants")
    filtered = [t for t in tenants if t.get("code") != "hyper"]
    if not filtered:
        await message.answer("هیچ مارکت فعالی برای مدیریت وجود ندارد.")
        return
    await state.set_state(ManageTenantStates.select)
    await message.answer(
        "مارکت مورد نظر را انتخاب کنید:",
        reply_markup=keyboards.tenants_keyboard(filtered),
    )


@router.callback_query(F.data == "tenant:back")
async def tenants_back(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(AdminStates.menu)
    await callback.answer()
    await callback.message.answer(
        "به منوی اصلی مدیریت هایپر برگشتید.",
        reply_markup=keyboards.hyper_admin_menu(),
    )


@router.callback_query(F.data == "tenant:list")
async def tenants_list(callback: types.CallbackQuery, state: FSMContext):
    client = await _get_client(state)
    tenants = await client.get("/tenants")
    filtered = [t for t in tenants if t.get("code") != "hyper"]
    await callback.answer()
    await state.set_state(ManageTenantStates.select)
    await callback.message.answer(
        "مارکت مورد نظر را انتخاب کنید:",
        reply_markup=keyboards.tenants_keyboard(filtered),
    )


@router.callback_query(F.data == "tenant:cancel")
async def tenants_cancel(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(AdminStates.menu)
    await callback.answer()
    await callback.message.answer(
        "به منوی اصلی مدیریت هایپر برگشتید.",
        reply_markup=keyboards.hyper_admin_menu(),
    )


@router.callback_query(F.data.startswith("tenant:"))
async def tenant_selected(callback: types.CallbackQuery, state: FSMContext):
    if callback.data in {"tenant:back", "tenant:list", "tenant:cancel"}:
        return

    tenant_id = callback.data.split(":", 1)[1]
    client = await _get_client(state)
    tenant = await client.get(f"/tenants/{tenant_id}")
    await _show_tenant_actions(callback, state, tenant)


@router.callback_query(F.data.startswith("tact:toggle:"))
async def tenant_toggle(callback: types.CallbackQuery, state: FSMContext):
    tenant_id = callback.data.split(":")[-1]
    client = await _get_client(state)
    tenant = await client.get(f"/tenants/{tenant_id}")
    new_status = "inactive" if tenant.get("status") == "active" else "active"
    await client.put(f"/tenants/{tenant_id}", {"status": new_status})
    refreshed = await client.get(f"/tenants/{tenant_id}")
    await callback.answer("وضعیت به‌روزرسانی شد.")
    await _show_tenant_actions(callback, state, refreshed, "وضعیت به‌روزرسانی شد.")


@router.callback_query(F.data.startswith("tact:notify:"))
async def tenant_notify(callback: types.CallbackQuery, state: FSMContext):
    tenant_id = callback.data.split(":")[-1]
    client = await _get_client(state)
    try:
        profile = await client.get(f"/tenants/{tenant_id}/profile", tenant_id=tenant_id)
    except Exception as exc:
        details = _short_exception(exc)
        await callback.answer(f"دریافت اطلاعات مدیران امکان‌پذیر نیست: {details}", show_alert=True)
        return
    manager_ids = _normalize_manager_ids(profile.get("manager_telegram_ids"))
    if not manager_ids:
        await callback.answer("حداقل یک مدیر ثبت نشده است. ابتدا مدیر اضافه کنید.", show_alert=True)
        return
    await state.update_data(
        selected_tenant_id=tenant_id,
        pending_notification_manager_ids=manager_ids,
    )
    await state.set_state(ManageTenantStates.awaiting_notification_message)
    await callback.answer()
    await callback.message.answer(
        "متن پیام مورد نظر را وارد کنید. برای لغو /cancel را ارسال کنید."
    )


@router.message(ManageTenantStates.awaiting_notification_message)
async def tenant_notification_message(message: types.Message, state: FSMContext):
    text = (message.text or "").strip()
    if not text:
        await message.answer("لطفاً متنی وارد کنید.")
        return
    await state.update_data(pending_notification_text=text)
    await state.set_state(ManageTenantStates.awaiting_notification_confirm)
    await message.answer(
        f"پیش‌نمایش پیام:\n{text}\nآیا مایل به ارسال هستید؟",
        reply_markup=keyboards.notification_confirm_keyboard(),
    )


@router.message(ManageTenantStates.awaiting_notification_message, Command("cancel"))
async def tenant_notification_message_cancel(message: types.Message, state: FSMContext):
    await state.set_state(ManageTenantStates.action)
    await message.answer("ارسال اعلان لغو شد.")


@router.callback_query(F.data == "notify:confirm")
async def tenant_notification_confirm(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    tenant_id = data.get("selected_tenant_id")
    text = data.get("pending_notification_text")
    manager_ids = data.get("pending_notification_manager_ids") or []
    if not (tenant_id and text and manager_ids):
        await callback.answer("مقادیر لازم برای ارسال پیام تکمیل نیست.", show_alert=True)
        return
    client = await _get_client(state)
    tenant = await client.get(f"/tenants/{tenant_id}")
    for manager_id in manager_ids:
        try:
            await callback.bot.send_message(
                manager_id,
                f"پیام مدیریت برای مارکت {tenant.get('title')}:\n\n{text}",
            )
        except Exception as exc:
            logger.warning("failed to notify manager %s: %s", manager_id, exc)
    await state.update_data(pending_notification_text=None, pending_notification_manager_ids=None)
    await state.set_state(ManageTenantStates.action)
    await callback.answer("پیام برای مدیران ارسال شد.", show_alert=True)
    await _show_tenant_actions(callback, state, tenant, "پیام برای مدیران ارسال شد.")


@router.callback_query(F.data == "notify:cancel")
async def tenant_notification_cancel(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    tenant_id = data.get("selected_tenant_id")
    client = await _get_client(state)
    tenant = await client.get(f"/tenants/{tenant_id}")
    await state.update_data(pending_notification_text=None, pending_notification_manager_ids=None)
    await state.set_state(ManageTenantStates.action)
    await callback.answer("ارسال اعلان لغو شد.", show_alert=True)
    await _show_tenant_actions(callback, state, tenant, "ارسال اعلان لغو شد.")


@router.callback_query(F.data.startswith("tact:add_admin:"))
async def tenant_add_admin(callback: types.CallbackQuery, state: FSMContext):
    tenant_id = callback.data.split(":")[-1]
    await state.update_data(selected_tenant_id=tenant_id)
    await state.set_state(ManageTenantStates.awaiting_manager_id)
    await callback.answer()
    await callback.message.answer(
        "شماره تلگرام مدیر جدید را وارد کنید (فقط اعداد). برای لغو /cancel انجام دهید."
    )


@router.message(ManageTenantStates.awaiting_manager_id)
async def tenant_manager_id_input(message: types.Message, state: FSMContext):
    data = await state.get_data()
    tenant_id = data.get("selected_tenant_id")
    if not tenant_id:
        await message.answer("ابتدا یک مارکت انتخاب کنید.")
        await state.set_state(ManageTenantStates.action)
        return
    parsed = _parse_manager_ids(message.text or "")
    if not parsed:
        await message.answer("لطفاً یک یا چند شناسه معتبر وارد کنید.")
        return
    client = await _get_client(state)
    try:
        profile = await client.get(f"/tenants/{tenant_id}/profile", tenant_id=tenant_id)
    except Exception as exc:
        details = _short_exception(exc)
        await message.answer(f"دریافت اطلاعات مدیران ممکن نبود: {details}")
        return
    existing = _normalize_manager_ids(profile.get("manager_telegram_ids") if profile else None)
    combined = existing + parsed
    seen: set[int] = set()
    unique: list[int] = []
    for value in combined:
        if value not in seen:
            seen.add(value)
            unique.append(value)
    try:
        await client.put(f"/tenants/{tenant_id}/profile", {"manager_telegram_ids": unique}, tenant_id=tenant_id)
    except Exception as exc:
        details = _short_exception(exc)
        await message.answer(f"ثبت مدیر جدید ناموفق بود: {details}")
        return
    await state.set_state(ManageTenantStates.action)
    tenant = await client.get(f"/tenants/{tenant_id}")
    await _show_tenant_actions(message, state, tenant, "مدیر جدید ثبت شد.")


@router.message(ManageTenantStates.awaiting_manager_id, Command("cancel"))
async def tenant_manager_id_cancel(message: types.Message, state: FSMContext):
    await state.set_state(ManageTenantStates.action)
    await message.answer("افزودن مدیر لغو شد.")


@router.callback_query(F.data.startswith("tact:edit_profile:"))
async def tenant_edit_profile(callback: types.CallbackQuery, state: FSMContext):
    tenant_id = callback.data.split(":")[-1]
    await state.update_data(selected_tenant_id=tenant_id)
    await state.set_state(ManageTenantStates.edit_profile)
    await callback.answer()
    await callback.message.answer(
        "برای ویرایش، فیلد مورد نظر را انتخاب کنید:",
        reply_markup=_profile_fields_keyboard(tenant_id),
    )


@router.callback_query(ManageTenantStates.edit_profile, F.data.startswith("tact:profile:"))
async def tenant_profile_field(callback: types.CallbackQuery, state: FSMContext):
    parts = callback.data.split(":")
    if len(parts) != 4:
        await callback.answer("فرمت داده نامعتبر است.", show_alert=True)
        return

    _, _, tenant_id, alias = parts
    field_key = ALIAS_TO_FIELD_KEY.get(alias)
    field = EDITABLE_FIELD_MAP.get(field_key) if field_key else None
    if not field:
        await callback.answer("این فیلد قابل ویرایش نیست.", show_alert=True)
        return

    await state.update_data(
        selected_tenant_id=tenant_id,
        editing_field=field_key,
        editing_scope=field["scope"],
        editing_label=field["label"],
    )
    await state.set_state(ManageTenantStates.awaiting_profile_value)
    await callback.answer()

    if field["type"] == "choice":
        await callback.message.answer(
            f"سطح جدید برای «{field['label']}» را انتخاب کنید:",
            reply_markup=keyboards.register_tier_keyboard(),
        )
    else:
        await callback.message.answer(
            f"مقدار جدید برای «{field['label']}» را وارد کنید.\n"
            "برای لغو /cancel را ارسال کنید."
        )


@router.callback_query(
    ManageTenantStates.awaiting_profile_value,
    F.data.startswith("reg:tier:"),
)
async def tenant_profile_set_tier(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    field_key = data.get("editing_field")
    tenant_id = data.get("selected_tenant_id")
    if field_key != "subscription_tier" or not tenant_id:
        await callback.answer()
        return
    tier = callback.data.split(":")[-1]
    await _apply_profile_update(callback, state, tenant_id, field_key, tier)


@router.message(ManageTenantStates.awaiting_profile_value)
async def tenant_profile_value(message: types.Message, state: FSMContext):
    data = await state.get_data()
    field_key = data.get("editing_field")
    tenant_id = data.get("selected_tenant_id")
    if not field_key or not tenant_id:
        await message.answer("خطایی رخ داد. دوباره تلاش کنید.")
        return

    value = (message.text or "").strip()
    if not value:
        await message.answer("مقداری وارد نشده است.")
        return

    if field_key == "manager_telegram_ids":
        parsed = _parse_manager_ids(value)
        if not parsed:
            await message.answer("لطفاً حداقل یک عدد معتبر وارد کنید.")
            return
        await _apply_profile_update(message, state, tenant_id, field_key, parsed)
    else:
        await _apply_profile_update(message, state, tenant_id, field_key, value)


@router.message(ManageTenantStates.awaiting_profile_value, Command("cancel"))
async def tenant_profile_cancel(message: types.Message, state: FSMContext):
    data = await state.get_data()
    tenant_id = data.get("selected_tenant_id")
    await state.update_data(editing_field=None, editing_scope=None, editing_label=None)

    if not tenant_id:
        await message.answer("هیچ مارکتی انتخاب نشده است.")
        await state.set_state(ManageTenantStates.action)
        return

    client = await _get_client(state)
    tenant = await client.get(f"/tenants/{tenant_id}")
    await _show_tenant_actions(message, state, tenant, "ویرایش لغو شد.")
