#!/usr/bin/env bash
set -euo pipefail

# ──────────────────────────────────────────────
#  Project Root Detection
# ──────────────────────────────────────────────
PROJECT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
if [[ ! -f "$PROJECT_DIR/app/main.py" ]]; then
    PROJECT_DIR="$(cd "$PROJECT_DIR/.." && pwd)"
fi
cd "$PROJECT_DIR"

# ──────────────────────────────────────────────
#  Virtualenv Activation
# ──────────────────────────────────────────────
if [[ -n "${BAZARYA_VENV:-}" ]]; then
    VENV_DIR="$BAZARYA_VENV"
elif [[ -n "${VIRTUAL_ENV:-}" ]]; then
    VENV_DIR="$VIRTUAL_ENV"
else
    VENV_DIR="$PROJECT_DIR/venv"
fi

if [[ ! -f "$VENV_DIR/bin/activate" ]]; then
    echo "❌ Virtual environment not found at: $VENV_DIR/bin/activate" >&2
    exit 1
fi

source "$VENV_DIR/bin/activate"

# ──────────────────────────────────────────────
#  Load .env Variables (if exists)
# ──────────────────────────────────────────────
if [[ -f .env ]]; then
    set -a
    source .env
    set +a
fi

# ──────────────────────────────────────────────
#  Stop Running Bots
# ──────────────────────────────────────────────
echo "🛑 Stopping Bazarya Python workers..."
for pattern in \
    "python -m app.bots.hyper.main" \
    "python -m app.bots.manager" \
    "python -m app.bots.store"
do
    pkill -f "$pattern" 2>/dev/null || true
done
sleep 1

# ──────────────────────────────────────────────
#  Reset DB + Alembic Upgrade
# ──────────────────────────────────────────────
echo "🗄  Resetting DB schema..."
python init_db.py
alembic upgrade head

# ──────────────────────────────────────────────
#  Run Bots
# ──────────────────────────────────────────────
echo "🚀 Starting Bots..."
mkdir -p logs/hyper_bot logs/store_bot

nohup python -m app.bots.hyper.main >> logs/hyper_bot/runner.log 2>&1 &
nohup python -m app.bots.manager   >> logs/store_bot/runner.log 2>&1 &

echo "✅ Done: DB reset + migrations complete + bots running."
