from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from sqlalchemy import select
from app.api.dependencies import get_db, get_tenant_context, require_roles
from app.models import Order
from app.services.cart import serialize_order, checkout as do_checkout
from app.schemas.order import CheckoutRequest
from app.services.payments import register_payment


router = APIRouter(prefix="/orders", tags=["orders"])


@router.get("/", response_model=list[dict])
def list_orders(
    status: str | None = None,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "store_staff", "hyper_admin")),
):
    stmt = select(Order).where(Order.tenant_id == tenant.id)
    if status:
        stmt = stmt.where(Order.status == status)
    stmt = stmt.order_by(Order.created_at.desc())
    orders = db.scalars(stmt).all()
    return [serialize_order(order) for order in orders]


@router.get("/{order_id}", response_model=dict)
def get_order(
    order_id: str,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "store_staff", "hyper_admin")),
):
    order = db.get(Order, order_id)
    if not order or str(order.tenant_id) != str(tenant.id):
        raise HTTPException(status_code=404, detail="order not found")
    return serialize_order(order)


@router.post("/{order_id}/checkout", response_model=dict)
def order_checkout(
    order_id: str,
    payload: CheckoutRequest,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "store_staff", "customer")),
):
    order = db.get(Order, order_id)
    if not order or str(order.tenant_id) != str(tenant.id):
        raise HTTPException(status_code=404, detail="order not found")
    if order.status not in {"cart", "saved"}:
        raise HTTPException(status_code=400, detail="order not in cart/saved state")
    order = do_checkout(db, order, payload)
    register_payment(db, order, order.total, payload.payment_method)
    return serialize_order(order)


@router.get("/my", response_model=list[dict])
def customer_orders(
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    user=Depends(require_roles("customer")),
):
    if not user.tenant_id or str(user.tenant_id) != str(tenant.id):
        raise HTTPException(status_code=403, detail="tenant mismatch")
    stmt = (
        select(Order)
        .where(Order.tenant_id == tenant.id, Order.customer_user_id == user.id)
        .order_by(Order.created_at.desc())
    )
    orders = db.scalars(stmt).all()
    return [serialize_order(order) for order in orders]


@router.get("/my/{order_id}", response_model=dict)
def customer_order_detail(
    order_id: str,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    user=Depends(require_roles("customer")),
):
    order = db.get(Order, order_id)
    if not order or str(order.tenant_id) != str(tenant.id) or str(order.customer_user_id) != str(user.id):
        raise HTTPException(status_code=404, detail="order not found")
    return serialize_order(order)
