from alembic import op
import sqlalchemy as sa

from app.db.types import GUID


TABLE_KWARGS = {"mysql_engine": "InnoDB", "mysql_charset": "utf8mb4"}


revision = "0003_sales"
down_revision = "0002_catalog"
branch_labels = None
depends_on = None


def upgrade():
    op.create_table(
        "store_customer",
        sa.Column("id", GUID(), primary_key=True),
        sa.Column("tenant_id", GUID(), sa.ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False),
        sa.Column("user_id", GUID(), sa.ForeignKey("app_user.id", ondelete="CASCADE"), nullable=False),
        sa.Column("first_name", sa.Text(), nullable=False),
        sa.Column("last_name", sa.Text(), nullable=False),
        sa.Column("phone", sa.Text(), nullable=False),
        sa.Column("postal_code", sa.Text()),
        sa.Column("address", sa.Text()),
        sa.Column("delivery_geo", sa.JSON()),
        sa.Column(
            "created_at",
            sa.TIMESTAMP(timezone=True),
            server_default=sa.text("CURRENT_TIMESTAMP"),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.TIMESTAMP(timezone=True),
            server_default=sa.text("CURRENT_TIMESTAMP"),
            nullable=False,
        ),
        sa.UniqueConstraint("tenant_id", "user_id", name="uq_store_customer_tenant_user"),
        **TABLE_KWARGS,
    )

    op.create_table(
        "order",
        sa.Column("id", GUID(), primary_key=True),
        sa.Column("tenant_id", GUID(), sa.ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False),
        sa.Column("customer_user_id", GUID(), sa.ForeignKey("app_user.id", ondelete="SET NULL")),
        sa.Column(
            "status",
            sa.Enum(
                "cart",
                "saved",
                "awaiting_payment",
                "paid",
                "canceled",
                name="order_status",
            ),
            nullable=False,
            server_default="cart",
        ),
        sa.Column(
            "payment_method",
            sa.Enum(
                "card2card",
                "online",
                "inperson",
                name="order_payment_method",
            ),
        ),
        sa.Column("total", sa.Numeric(12, 2), nullable=False, server_default="0"),
        sa.Column(
            "created_at",
            sa.TIMESTAMP(timezone=True),
            server_default=sa.text("CURRENT_TIMESTAMP"),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.TIMESTAMP(timezone=True),
            server_default=sa.text("CURRENT_TIMESTAMP"),
            nullable=False,
        ),
        sa.CheckConstraint("status IN ('cart','saved','awaiting_payment','paid','canceled')"),
        sa.CheckConstraint(
            "payment_method IS NULL OR payment_method IN ('card2card','online','inperson')"
        ),
        **TABLE_KWARGS,
    )
    op.create_index("idx_order_tenant_status", "order", ["tenant_id", "status"])

    op.create_table(
        "order_item",
        sa.Column("id", GUID(), primary_key=True),
        sa.Column("tenant_id", GUID(), sa.ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False),
        sa.Column("order_id", GUID(), sa.ForeignKey("order.id", ondelete="CASCADE"), nullable=False),
        sa.Column("product_id", GUID(), sa.ForeignKey("product.id", ondelete="RESTRICT"), nullable=False),
        sa.Column("qty", sa.Integer(), nullable=False),
        sa.Column("unit_price_snapshot", sa.Numeric(12, 2), nullable=False),
        sa.Column("discount_percent_snapshot", sa.Integer(), nullable=False, server_default="0"),
        sa.Column("final_price_snapshot", sa.Numeric(12, 2), nullable=False),
        sa.Column("chosen_specs_json", sa.JSON()),
        sa.Column(
            "created_at",
            sa.TIMESTAMP(timezone=True),
            server_default=sa.text("CURRENT_TIMESTAMP"),
            nullable=False,
        ),
        sa.CheckConstraint("qty > 0"),
        sa.CheckConstraint("discount_percent_snapshot BETWEEN 0 AND 90"),
        **TABLE_KWARGS,
    )
    op.create_index("idx_order_item_order", "order_item", ["order_id"])

    op.create_table(
        "payment",
        sa.Column("id", GUID(), primary_key=True),
        sa.Column("tenant_id", GUID(), sa.ForeignKey("tenant.id", ondelete="CASCADE"), nullable=False),
        sa.Column("order_id", GUID(), sa.ForeignKey("order.id", ondelete="CASCADE"), nullable=False),
        sa.Column(
            "method",
            sa.Enum(
                "card2card",
                "online",
                "inperson",
                name="payment_method",
            ),
            nullable=False,
        ),
        sa.Column("amount", sa.Numeric(12, 2), nullable=False),
        sa.Column("gateway_tx_id", sa.Text()),
        sa.Column("card2card_note", sa.Text()),
        sa.Column(
            "status",
            sa.Enum(
                "pending",
                "confirmed",
                "failed",
                name="payment_status",
            ),
            nullable=False,
            server_default="pending",
        ),
        sa.Column(
            "created_at",
            sa.TIMESTAMP(timezone=True),
            server_default=sa.text("CURRENT_TIMESTAMP"),
            nullable=False,
        ),
        sa.CheckConstraint("method IN ('card2card','online','inperson')"),
        sa.CheckConstraint("status IN ('pending','confirmed','failed')"),
        sa.CheckConstraint("amount >= 0"),
        **TABLE_KWARGS,
    )


def downgrade():
    op.drop_table("payment")
    op.drop_index("idx_order_item_order", table_name="order_item")
    op.drop_table("order_item")
    op.drop_index("idx_order_tenant_status", table_name="order")
    op.drop_table("order")
    op.drop_table("store_customer")
