from fastapi import APIRouter, Depends, HTTPException, Header
from sqlalchemy.orm import Session
from sqlalchemy import select
from app.db.session import get_db
from app.models import AppUser
from app.core.security import verify_password, create_access_token, hash_password
from app.schemas.auth import LoginRequest, TokenResponse
from app.api.dependencies import get_tenant_context


router = APIRouter(prefix="/auth", tags=["auth"])


@router.post("/login", response_model=TokenResponse)
def login(payload: LoginRequest, db: Session = Depends(get_db)):
    stmt = select(AppUser).where(AppUser.login_username == payload.username)
    user = db.scalars(stmt).first()
    if not user or not user.password_hash or not verify_password(payload.password, user.password_hash):
        raise HTTPException(status_code=401, detail="invalid credentials")

    if payload.telegram_user_id:
        if user.telegram_user_id and user.telegram_user_id != payload.telegram_user_id:
            raise HTTPException(status_code=403, detail="telegram binding mismatch")
        if not user.telegram_user_id:
            user.telegram_user_id = payload.telegram_user_id
            db.add(user)
            db.commit()
            db.refresh(user)

    token = create_access_token(str(user.id), user.role, str(user.tenant_id) if user.tenant_id else None)
    return TokenResponse(
        token=token,
        role=user.role,
        tenant_id=str(user.tenant_id) if user.tenant_id else None,
        telegram_user_id=user.telegram_user_id,
    )


@router.post('/telegram-login', response_model=TokenResponse)
def telegram_login(
    x_telegram_user: str = Header(..., alias='X-Telegram-User'),
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
):
    try:
        tg_id = int(x_telegram_user)
    except ValueError:
        raise HTTPException(status_code=400, detail='invalid telegram header')
    user = db.scalars(
        select(AppUser).where(AppUser.tenant_id == tenant.id, AppUser.telegram_user_id == tg_id)
    ).first()
    if not user:
        user = AppUser(
            tenant_id=tenant.id,
            telegram_user_id=tg_id,
            login_username=None,
            password_hash=hash_password(str(tg_id)),
            role='customer',
            status='active',
        )
        db.add(user)
        db.commit()
        db.refresh(user)
    token = create_access_token(str(user.id), user.role, str(user.tenant_id))
    return TokenResponse(token=token, role=user.role, tenant_id=str(user.tenant_id), telegram_user_id=user.telegram_user_id)
