from __future__ import annotations
import httpx
from typing import Any
from app.core.config import get_settings
from app.core.logging import get_logger


logger = get_logger(__name__)
settings = get_settings()


async def verify_bot_token(token: str) -> dict[str, Any]:
    if not token:
        raise ValueError("telegram token is required")
    if not settings.TELEGRAM_VERIFY_TOKENS:
        logger.info("Skipping telegram token verification", extra={"token_prefix": token[:6]})
        return {"username": None}
    url = f"{settings.TELEGRAM_API_BASE}/bot{token}/getMe"
    async with httpx.AsyncClient(timeout=10) as client:
        try:
            resp = await client.get(url)
            resp.raise_for_status()
        except httpx.HTTPError as exc:
            logger.warning("Telegram token verification failed", extra={"status": getattr(exc.response, 'status_code', None)})
            raise ValueError("invalid telegram token") from exc
        data = resp.json()
        if not data.get("ok"):
            logger.warning("Telegram token verification failed", extra={"response": data})
            raise ValueError("invalid telegram token")
        return data.get("result", data)

