import json
import logging
from logging.config import dictConfig
from pathlib import Path
from app.core.config import get_settings
from app.core.context import get_correlation_id


class JsonFormatter(logging.Formatter):
    def format(self, record: logging.LogRecord) -> str:  # type: ignore[override]
        data = {
            "level": record.levelname,
            "logger": record.name,
            "message": record.getMessage(),
            "time": self.formatTime(record, self.datefmt),
        }
        correlation_id = get_correlation_id()
        if correlation_id:
            data["correlation_id"] = correlation_id
        if record.exc_info:
            data["exc_info"] = self.formatException(record.exc_info)
        return json.dumps(data, ensure_ascii=False)


def setup_logging() -> None:
    settings = get_settings()
    base_log_dir = Path("logs")
    hyper_log = base_log_dir / "hyper_bot" / "hyper.log"
    store_log = base_log_dir / "store_bot" / "store.log"
    hyper_log.parent.mkdir(parents=True, exist_ok=True)
    store_log.parent.mkdir(parents=True, exist_ok=True)

    dictConfig(
        {
            "version": 1,
            "disable_existing_loggers": False,
            "formatters": {
                "json": {
                    "()": JsonFormatter,
                }
            },
            "handlers": {
                "console": {
                    "class": "logging.StreamHandler",
                    "formatter": "json",
                    "level": settings.LOG_LEVEL,
                },
                "hyper_file": {
                    "class": "logging.FileHandler",
                    "formatter": "json",
                    "filename": str(hyper_log),
                    "encoding": "utf-8",
                    "level": settings.LOG_LEVEL,
                },
                "store_file": {
                    "class": "logging.FileHandler",
                    "formatter": "json",
                    "filename": str(store_log),
                    "encoding": "utf-8",
                    "level": settings.LOG_LEVEL,
                },
            },
            "root": {
                "handlers": ["console"],
                "level": settings.LOG_LEVEL,
            },
            "loggers": {
                "app.bots.hyper": {
                    "handlers": ["console", "hyper_file"],
                    "level": settings.LOG_LEVEL,
                    "propagate": False,
                },
                "app.bots.store": {
                    "handlers": ["console", "store_file"],
                    "level": settings.LOG_LEVEL,
                    "propagate": False,
                },
            },
        }
    )


def get_logger(name: str) -> logging.Logger:
    return logging.getLogger(name)
