# -*- coding: utf-8 -*-
from aiogram.types import ReplyKeyboardMarkup, KeyboardButton, InlineKeyboardMarkup, InlineKeyboardButton


def main_menu() -> ReplyKeyboardMarkup:
    """
    منوی اصلی برای کاربران عادی (غیر مدیر هایپر).
    - دکمه «📮 درخواست خرید حساب» برای شروع فرآیند خرید
    - دکمه «ℹ️ پشتیبانی» برای ارتباط با پشتیبانی
    - دکمه «📘 راهنما» برای نمایش توضیحات
    - دکمه «🔐 ورود مدیر هایپر» برای ورود ادمین هایپر
    """
    keyboard = [
        [KeyboardButton(text="📮 درخواست خرید حساب")],
        [KeyboardButton(text="ℹ️ پشتیبانی"), KeyboardButton(text="📘 راهنما")],
        [KeyboardButton(text="🔐 ورود مدیر هایپر")],
    ]
    return ReplyKeyboardMarkup(
        keyboard=keyboard,
        resize_keyboard=True,
    )


def hyper_admin_menu() -> ReplyKeyboardMarkup:
    """
    منوی اصلی مدیر هایپر.
    """
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="➕ ثبت مارکت جدید"), KeyboardButton(text="📂 مدیریت مارکت‌ها")],
            [KeyboardButton(text="⚙️ تنظیمات پایه"), KeyboardButton(text="↩️ خروج")],
        ],
        resize_keyboard=True,
    )


def register_input_keyboard(step_key: str) -> InlineKeyboardMarkup:
    """
    (در نسخه قبلی استفاده می‌شد، الان دیگر در جریان جدید ثبت مارکت استفاده نمی‌شود،
    ولی برای سازگاری نگه داشته شده است. اگر در آینده خواستی می‌توانی حذفش کنی.)
    """
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="✏️ ارسال مقدار", callback_data=f"reg:input:{step_key}")],
            [
                InlineKeyboardButton(text="⬅️ بازگشت", callback_data="reg:back"),
                InlineKeyboardButton(text="❌ لغو", callback_data="reg:cancel"),
            ],
        ]
    )


def register_tier_keyboard() -> InlineKeyboardMarkup:
    """
    انتخاب سطح/پلن اشتراک مارکت در هنگام ثبت.
    """
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [
                InlineKeyboardButton(text="Base", callback_data="reg:tier:base"),
                InlineKeyboardButton(text="Silver", callback_data="reg:tier:silver"),
                InlineKeyboardButton(text="Gold", callback_data="reg:tier:gold"),
            ],
            [
                InlineKeyboardButton(text="⬅️ بازگشت", callback_data="reg:back"),
                InlineKeyboardButton(text="❌ لغو", callback_data="reg:cancel"),
            ],
        ]
    )


def register_confirm_keyboard() -> InlineKeyboardMarkup:
    """
    کیبورد تایید نهایی ثبت مارکت.
    """
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="✅ تایید نهایی", callback_data="reg:confirm")],
            [
                InlineKeyboardButton(text="⬅️ بازگشت", callback_data="reg:back"),
                InlineKeyboardButton(text="❌ لغو", callback_data="reg:cancel"),
            ],
        ]
    )


STATUS_LABEL = {"active": "فعال", "inactive": "غیرفعال"}


def tenants_keyboard(tenants: list[dict]) -> InlineKeyboardMarkup:
    """
    لیست مارکت‌ها برای انتخاب در بخش مدیریت مارکت‌ها.
    """
    rows = []
    for t in tenants:
        status = STATUS_LABEL.get(t.get("status"), "نامشخص")
        rows.append(
            [
                InlineKeyboardButton(
                    text=f"{t['title']} ({t['code']}) · {status}",
                    callback_data=f"tenant:{t['id']}",
                )
            ]
        )
    rows.append([InlineKeyboardButton(text="⬅️ بازگشت", callback_data="tenant:back")])
    return InlineKeyboardMarkup(inline_keyboard=rows)


def tenant_actions_keyboard(tenant_id: str, is_active: bool) -> InlineKeyboardMarkup:
    """
    کیبورد اکشن‌های مربوط به یک مارکت:
    - فعال/غیرفعال‌سازی
    - افزودن مدیر
    - ویرایش پروفایل
    - ارسال اعلان
    - بازگشت/لغو
    """
    rows = [
        [
            InlineKeyboardButton(
                text=("⏸️ غیرفعال‌سازی" if is_active else "▶️ فعال‌سازی"),
                callback_data=f"tact:toggle:{tenant_id}",
            )
        ],
        [InlineKeyboardButton(text="👤 افزودن مدیر", callback_data=f"tact:add_admin:{tenant_id}")],
        [InlineKeyboardButton(text="📝 ویرایش پروفایل", callback_data=f"tact:edit_profile:{tenant_id}")],
        [InlineKeyboardButton(text="📣 ارسال اعلان", callback_data=f"tact:notify:{tenant_id}")],
        [InlineKeyboardButton(text="⬅️ بازگشت", callback_data="tenant:list")],
        [InlineKeyboardButton(text="❌ لغو", callback_data="tenant:cancel")],
    ]
    return InlineKeyboardMarkup(inline_keyboard=rows)


def notification_confirm_keyboard() -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="✉️ ارسال پیام", callback_data="notify:confirm")],
            [InlineKeyboardButton(text="❌ لغو", callback_data="notify:cancel")],
        ]
    )


def base_settings_keyboard() -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="ℹ️ درباره ما", callback_data="base:about")],
            [InlineKeyboardButton(text="📘 راهنما", callback_data="base:guide")],
            [InlineKeyboardButton(text="⬅️ بازگشت", callback_data="base:cancel")],
        ]
    )


# ------------------ کیبوردهای مربوط به خرید حساب (Lead) ------------------


def lead_plan_keyboard() -> InlineKeyboardMarkup:
    """
    انتخاب پلن توسط کاربر برای خرید حساب.
    """
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [
                InlineKeyboardButton(text="پلن پایه", callback_data="lead:plan:base"),
                InlineKeyboardButton(text="پلن نقره‌ای", callback_data="lead:plan:silver"),
            ],
            [
                InlineKeyboardButton(text="پلن طلایی", callback_data="lead:plan:gold"),
            ],
            [
                InlineKeyboardButton(text="❌ لغو", callback_data="lead:cancel"),
            ],
        ]
    )


def lead_wait_payment_keyboard() -> InlineKeyboardMarkup:
    """
    زمانی که کاربر اطلاعات پرداخت را ارسال کرده و باید منتظر تایید بماند.
    """
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [
                InlineKeyboardButton(
                    text="⏳ در انتظار تایید پرداخت",
                    callback_data="lead:wait_payment:noop",  # صرفاً برای نمایش؛ هندلر خاصی روی آن نیست
                )
            ],
            [
                InlineKeyboardButton(text="🛑 لغو و بازگشت به منو", callback_data="lead:cancel"),
            ],
        ]
    )
