from __future__ import annotations
import secrets
from sqlalchemy.orm import Session
from sqlalchemy import select
from app.models import StoreCustomer, AppUser
from app.schemas.customer import CustomerCreate, CustomerUpdate
from app.utils.validators import ensure_iran_phone
from app.core.security import hash_password


def _get_or_create_customer_user(db: Session, tenant_id: str, phone: str) -> AppUser:
    # try find by phone
    user = db.scalars(select(AppUser).where(AppUser.tenant_id == tenant_id, AppUser.phone == phone)).first()
    if user:
        return user
    username = f"cust_{secrets.token_hex(4)}"
    password = secrets.token_urlsafe(8)
    user = AppUser(
        tenant_id=tenant_id,
        login_username=username,
        password_hash=hash_password(password),
        role="customer",
        phone=phone,
        status="active",
    )
    db.add(user)
    db.flush()
    return user


def list_customers(db: Session, tenant_id: str) -> list[StoreCustomer]:
    return list(db.scalars(select(StoreCustomer).where(StoreCustomer.tenant_id == tenant_id)))


def create_customer(db: Session, tenant_id: str, payload: CustomerCreate) -> StoreCustomer:
    ensure_iran_phone(payload.phone)
    user = _get_or_create_customer_user(db, tenant_id, payload.phone)
    sc = StoreCustomer(
        tenant_id=tenant_id,
        user_id=user.id,
        first_name=payload.first_name,
        last_name=payload.last_name,
        phone=payload.phone,
        postal_code=payload.postal_code,
        address=payload.address,
        delivery_geo=payload.delivery_geo,
    )
    db.add(sc)
    db.commit()
    db.refresh(sc)
    return sc


def update_customer(db: Session, customer: StoreCustomer, payload: CustomerUpdate) -> StoreCustomer:
    data = payload.dict(exclude_unset=True)
    if "phone" in data:
        ensure_iran_phone(data["phone"])
        customer.phone = data["phone"]
    if "first_name" in data:
        customer.first_name = data["first_name"]
    if "last_name" in data:
        customer.last_name = data["last_name"]
    if "postal_code" in data:
        customer.postal_code = data["postal_code"]
    if "address" in data:
        customer.address = data["address"]
    if "delivery_geo" in data:
        customer.delivery_geo = data["delivery_geo"]
    db.add(customer)
    db.commit()
    db.refresh(customer)
    return customer
