from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from app.api.dependencies import get_db, get_tenant_context, require_roles
from app.schemas.customer import CustomerCreate, CustomerUpdate, CustomerOut
from app.services.customers import list_customers, create_customer, update_customer
from app.models import StoreCustomer


router = APIRouter(prefix="/customers", tags=["customers"])


@router.get("/", response_model=list[CustomerOut])
def customers_list(
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "store_staff")),
):
    records = list_customers(db, str(tenant.id))
    return [
        CustomerOut(
            id=r.id,
            user_id=r.user_id,
            first_name=r.first_name,
            last_name=r.last_name,
            phone=r.phone,
            postal_code=r.postal_code,
            address=r.address,
        )
        for r in records
    ]


@router.post("/", response_model=CustomerOut)
def customers_create(
    payload: CustomerCreate,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "store_staff")),
):
    c = create_customer(db, str(tenant.id), payload)
    return CustomerOut(
        id=c.id,
        user_id=c.user_id,
        first_name=c.first_name,
        last_name=c.last_name,
        phone=c.phone,
        postal_code=c.postal_code,
        address=c.address,
    )


@router.put("/{customer_id}", response_model=CustomerOut)
def customers_update(
    customer_id: str,
    payload: CustomerUpdate,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "store_staff")),
):
    c = db.get(StoreCustomer, customer_id)
    if not c or str(c.tenant_id) != str(tenant.id):
        raise HTTPException(status_code=404, detail="customer not found")
    c = update_customer(db, c, payload)
    return CustomerOut(
        id=c.id,
        user_id=c.user_id,
        first_name=c.first_name,
        last_name=c.last_name,
        phone=c.phone,
        postal_code=c.postal_code,
        address=c.address,
    )
