import io
from fastapi import APIRouter, Depends, Query
from fastapi.responses import StreamingResponse
from sqlalchemy.orm import Session
from openpyxl import Workbook
from app.api.dependencies import get_db, get_tenant_context, require_roles
from app.schemas.report import ReportFilters
from app.services.reports import build_orders_report


router = APIRouter(prefix="/reports", tags=["reports"])


@router.get("/orders")
def orders_report(
    date_from: str,
    date_to: str,
    export: str | None = Query(default=None, description="set to 'xlsx' for spreadsheet output"),
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "hyper_admin")),
):
    filters = ReportFilters(date_from=date_from, date_to=date_to)
    report = build_orders_report(db, str(tenant.id), filters)
    if export == "xlsx":
        wb = Workbook()
        ws = wb.active
        ws.title = "orders"
        ws.append(["Order ID", "Status", "Payment", "Total", "Created At"])
        for row in report.rows:
            ws.append([
                row.order_id,
                row.status,
                row.payment_method,
                row.total,
                row.created_at.isoformat(),
            ])
        ws.append(["", "", "جمع", report.total_amount])
        stream = io.BytesIO()
        wb.save(stream)
        stream.seek(0)
        filename = "orders-report.xlsx"
        return StreamingResponse(
            stream,
            media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
            headers={"Content-Disposition": f"attachment; filename={filename}"},
        )
    return report.dict()
