from __future__ import annotations
from fastapi import Depends, Header, HTTPException, status
from sqlalchemy.orm import Session
from sqlalchemy import select
from app.db.session import get_db
from app.models import AppUser, Tenant
from app.core.security import decode_access_token


def _get_authorization_token(authorization: str | None) -> str:
    if not authorization or not authorization.lower().startswith("bearer "):
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="missing bearer token")
    return authorization.split(" ", 1)[1]


def _coerce_telegram_id(raw: str | None) -> int | None:
    if raw is None:
        return None
    try:
        value = int(raw)
    except ValueError as exc:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail="invalid telegram header") from exc
    if value <= 0:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail="invalid telegram header")
    return value


def get_current_user(
    authorization: str | None = Header(default=None, alias="Authorization"),
    telegram_header: str | None = Header(default=None, alias="X-Telegram-User"),
    db: Session = Depends(get_db),
) -> AppUser:
    token = _get_authorization_token(authorization)
    payload = decode_access_token(token)
    stmt = select(AppUser).where(AppUser.id == payload["sub"])
    user = db.scalars(stmt).first()
    if not user or user.status != "active":
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="invalid session")

    telegram_id = _coerce_telegram_id(telegram_header)
    if telegram_id:
        if user.telegram_user_id and user.telegram_user_id != telegram_id:
            raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="telegram binding mismatch")
        if not user.telegram_user_id:
            user.telegram_user_id = telegram_id
            db.add(user)
            db.commit()
            db.refresh(user)
    return user


def require_roles(*roles: str):
    def _role_dependency(user: AppUser = Depends(get_current_user)) -> AppUser:
        if user.role not in roles:
            raise HTTPException(status_code=status.HTTP_403_FORBIDDEN, detail="insufficient role")
        return user

    return _role_dependency


def get_tenant_context(
    x_tenant_id: str = Header(..., alias="X-Tenant-Id"),
    db: Session = Depends(get_db),
) -> Tenant:
    tenant = db.get(Tenant, x_tenant_id)
    if not tenant or tenant.status != "active":
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="tenant not found")
    return tenant
