from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from app.api.dependencies import get_db, get_tenant_context, get_current_user, require_roles
from app.schemas.order import CartItemIn, CheckoutRequest
from app.services import cart as cart_service
from app.services.payments import register_payment


router = APIRouter(prefix="/cart", tags=["cart"])


def _assert_membership(user, tenant):
    if user.tenant_id and str(user.tenant_id) != str(tenant.id):
        raise HTTPException(status_code=403, detail="tenant mismatch")


@router.post("/start", response_model=dict)
def start_cart(
    tenant=Depends(get_tenant_context),
    user=Depends(require_roles("store_admin", "store_staff", "customer")),
    db: Session = Depends(get_db),
):
    _assert_membership(user, tenant)
    order = cart_service.get_or_create_cart(db, str(tenant.id), str(user.id))
    return cart_service.serialize_order(order)


@router.post("/items", response_model=dict)
def add_item(
    payload: CartItemIn,
    tenant=Depends(get_tenant_context),
    user=Depends(require_roles("store_admin", "store_staff", "customer")),
    db: Session = Depends(get_db),
):
    _assert_membership(user, tenant)
    order = cart_service.get_or_create_cart(db, str(tenant.id), str(user.id))
    order = cart_service.add_item(db, order, payload)
    return cart_service.serialize_order(order)


@router.delete("/items/{item_id}", response_model=dict)
def remove_item(
    item_id: str,
    tenant=Depends(get_tenant_context),
    user=Depends(require_roles("store_admin", "store_staff", "customer")),
    db: Session = Depends(get_db),
):
    _assert_membership(user, tenant)
    order = cart_service.get_or_create_cart(db, str(tenant.id), str(user.id))
    order = cart_service.remove_item(db, order, item_id)
    return cart_service.serialize_order(order)


@router.post("/save", response_model=dict)
def save_cart(
    tenant=Depends(get_tenant_context),
    user=Depends(require_roles("store_admin", "store_staff", "customer")),
    db: Session = Depends(get_db),
):
    _assert_membership(user, tenant)
    order = cart_service.get_or_create_cart(db, str(tenant.id), str(user.id))
    order = cart_service.save_cart(db, order)
    return cart_service.serialize_order(order)


@router.post("/cancel")
def cancel_cart(
    tenant=Depends(get_tenant_context),
    user=Depends(require_roles("store_admin", "store_staff", "customer")),
    db: Session = Depends(get_db),
):
    _assert_membership(user, tenant)
    order = cart_service.get_or_create_cart(db, str(tenant.id), str(user.id))
    cart_service.cancel_cart(db, order)
    return {"cancelled": True}


@router.post("/checkout", response_model=dict)
def checkout(
    payload: CheckoutRequest,
    tenant=Depends(get_tenant_context),
    user=Depends(require_roles("store_admin", "store_staff", "customer")),
    db: Session = Depends(get_db),
):
    _assert_membership(user, tenant)
    order = cart_service.get_or_create_cart(db, str(tenant.id), str(user.id))
    order = cart_service.checkout(db, order, payload)
    register_payment(db, order, order.total, payload.payment_method)
    return cart_service.serialize_order(order)
