from aiogram.types import (
    InlineKeyboardButton,
    InlineKeyboardMarkup,
    KeyboardButton,
    ReplyKeyboardMarkup,
    ReplyKeyboardRemove,
)


def landing_menu(show_admin_entry: bool = True) -> ReplyKeyboardMarkup:
    rows = [
        [KeyboardButton(text="📢 مشاهده محصولات")],
        [KeyboardButton(text="👤 حساب کاربری")],
    ]
    if show_admin_entry:
        rows.append([KeyboardButton(text="🔐 مدیریت مارکت")])
    return ReplyKeyboardMarkup(keyboard=rows, resize_keyboard=True)


def admin_main_menu() -> ReplyKeyboardMarkup:
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="📦 محصولات"), KeyboardButton(text="📁 دسته‌بندی‌ها")],
            [KeyboardButton(text="🧾 سفارش‌ها"), KeyboardButton(text="📣 کانال فروشگاه")],
            [KeyboardButton(text="↩️ خروج")],
        ],
        resize_keyboard=True,
    )


def admin_products_menu() -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="➕ ثبت محصول جدید", callback_data="admin:product:new")],
            [
                InlineKeyboardButton(text="📃 محصولات فعال", callback_data="admin:product:list:active"),
                InlineKeyboardButton(text="🚫 محصولات غیرفعال", callback_data="admin:product:list:inactive"),
            ],
            [InlineKeyboardButton(text="⬅️ بازگشت", callback_data="admin:back:main")],
            [InlineKeyboardButton(text="❌ لغو", callback_data="admin:cancel")],
        ]
    )


def admin_categories_menu() -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="➕ افزودن دسته", callback_data="admin:cat:new")],
            [InlineKeyboardButton(text="📋 لیست دسته‌ها", callback_data="admin:cat:list")],
            [InlineKeyboardButton(text="⬅️ بازگشت", callback_data="admin:back:main")],
            [InlineKeyboardButton(text="❌ لغو", callback_data="admin:cancel")],
        ]
    )


def admin_orders_menu() -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="🟡 سفارش‌های ذخیره‌شده", callback_data="admin:orders:saved")],
            [InlineKeyboardButton(text="🟢 سفارش‌های پرداخت‌شده", callback_data="admin:orders:paid")],
            [InlineKeyboardButton(text="⬅️ بازگشت", callback_data="admin:back:main")],
            [InlineKeyboardButton(text="❌ لغو", callback_data="admin:cancel")],
        ]
    )


def admin_channel_menu() -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="🔗 تنظیم channel_id", callback_data="admin:channel:set")],
            [InlineKeyboardButton(text="📤 ارسال تست", callback_data="admin:channel:test")],
            [InlineKeyboardButton(text="⬅️ بازگشت", callback_data="admin:back:main")],
            [InlineKeyboardButton(text="❌ لغو", callback_data="admin:cancel")],
        ]
    )


def wizard_back_cancel(callback_prefix: str) -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [
                InlineKeyboardButton(text="⬅️ بازگشت", callback_data=f"{callback_prefix}:back"),
                InlineKeyboardButton(text="❌ لغو", callback_data=f"{callback_prefix}:cancel"),
            ]
        ]
    )


def customer_account_menu(logged_in: bool) -> InlineKeyboardMarkup:
    if logged_in:
        rows = [
            [InlineKeyboardButton(text="🛒 سبد خرید", callback_data="acct:cart")],
            [InlineKeyboardButton(text="🧾 سوابق خرید", callback_data="acct:history")],
            [InlineKeyboardButton(text="📦 پیگیری سفارش", callback_data="acct:track")],
            [InlineKeyboardButton(text="↩️ بازگشت", callback_data="acct:back")],
        ]
    else:
        rows = [
            [InlineKeyboardButton(text="🔑 ورود", callback_data="acct:login")],
            [InlineKeyboardButton(text="🆕 ساخت حساب", callback_data="acct:register")],
            [InlineKeyboardButton(text="↩️ بازگشت", callback_data="acct:back")],
        ]
    return InlineKeyboardMarkup(inline_keyboard=rows)


def customer_main_menu(logged_in: bool, session_active: bool = False) -> InlineKeyboardMarkup:
    rows = [[InlineKeyboardButton(text="🚪 ورود به کانال فروشگاه", callback_data="cust:enter_channel")]]
    if session_active:
        rows.append([InlineKeyboardButton(text="🔚 پایان بازدید", callback_data="cust:end_visit")])
    rows.append([InlineKeyboardButton(text="📂 مشاهده دسته‌ها", callback_data="cust:browse")])
    if logged_in:
        rows.append([InlineKeyboardButton(text="🛒 سبد خرید من", callback_data="cust:cart")])
    else:
        rows.append([InlineKeyboardButton(text="🔐 ورود سریع با تلگرام", callback_data="cust:login")])
    rows.append([InlineKeyboardButton(text="↩️ بازگشت", callback_data="cust:back")])
    return InlineKeyboardMarkup(inline_keyboard=rows)


def account_contact_keyboard() -> ReplyKeyboardMarkup:
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="📱 ارسال شماره من", request_contact=True)],
            [KeyboardButton(text="لغو")],
        ],
        resize_keyboard=True,
        one_time_keyboard=True,
    )


def customer_categories_kb(categories: list[dict], logged_in: bool) -> InlineKeyboardMarkup:
    rows = [[InlineKeyboardButton(text=cat["name"], callback_data=f"cust:cat:{cat['id']}")] for cat in categories]
    if logged_in:
        rows.append([InlineKeyboardButton(text="🛒 سبد خرید", callback_data="cust:cart")])
    rows.append([InlineKeyboardButton(text="↩️ بازگشت به منو", callback_data="cust:menu")])
    if not rows:
        rows = [[InlineKeyboardButton(text="↩️ بازگشت به منو", callback_data="cust:menu")]]
    return InlineKeyboardMarkup(inline_keyboard=rows)


def customer_products_kb(products: list[dict], logged_in: bool) -> InlineKeyboardMarkup:
    rows: list[list[InlineKeyboardButton]] = []
    for prod in products:
        label = prod.get("title") or prod.get("name")
        rows.append([InlineKeyboardButton(text=label, callback_data=f"cust:add:{prod['id']}")])
    if logged_in:
        rows.append([InlineKeyboardButton(text="🛒 مشاهده سبد خرید", callback_data="cust:cart")])
    else:
        rows.append([InlineKeyboardButton(text="🔐 ورود برای خرید", callback_data="cust:login")])
    rows.append([InlineKeyboardButton(text="⬅️ بازگشت به دسته‌ها", callback_data="cust:browse")])
    return InlineKeyboardMarkup(inline_keyboard=rows)


def customer_cart_kb() -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="✅ نهایی‌سازی سفارش", callback_data="cust:checkout")],
            [InlineKeyboardButton(text="✏️ ویرایش سبد", callback_data="cust:edit_cart")],
            [InlineKeyboardButton(text="🗑 حذف سبد", callback_data="cust:cart_cancel")],
            [InlineKeyboardButton(text="⬅️ ادامه خرید", callback_data="cust:browse")],
        ]
    )


def product_categories_wizard(categories: list[dict], selected: list[str]) -> InlineKeyboardMarkup:
    rows: list[list[InlineKeyboardButton]] = []
    selected_set = set(selected or [])
    for cat in categories:
        cid = cat["id"]
        mark = "✅" if cid in selected_set else "⬜️"
        rows.append(
            [
                InlineKeyboardButton(
                    text=f"{mark} {cat['name']}",
                    callback_data=f"prod:cat:toggle:{cid}",
                )
            ]
        )
    rows.append(
        [
            InlineKeyboardButton(text="✅ تایید انتخاب", callback_data="prod:cat:done"),
            InlineKeyboardButton(text="⛔ بدون دسته", callback_data="prod:cat:skip"),
        ]
    )
    rows.append(
        [
            InlineKeyboardButton(text="⬅️ بازگشت", callback_data="prod:back"),
            InlineKeyboardButton(text="❌ لغو", callback_data="prod:cancel"),
        ]
    )
    return InlineKeyboardMarkup(inline_keyboard=rows)


def product_channel_keyboard(product_id: str, checkout_url: str | None = None) -> InlineKeyboardMarkup:
    rows = [
        [
            InlineKeyboardButton(text="❤️ پسند", callback_data=f"chan:like:{product_id}"),
            InlineKeyboardButton(text="➕ افزودن به سبد", callback_data=f"chan:add:{product_id}"),
        ]
    ]
    if checkout_url:
        rows.append([InlineKeyboardButton(text="🛒 تکمیل خرید", url=checkout_url)])
    else:
        rows.append([InlineKeyboardButton(text="🛒 تکمیل خرید", callback_data=f"chan:checkout:{product_id}")])
    return InlineKeyboardMarkup(inline_keyboard=rows)
