import os
from typing import List, Optional, Tuple, Dict, Any

import pandas as pd
import pymysql


MYSQL_HOST = os.getenv("MYSQL_HOST", "localhost")
MYSQL_PORT = int(os.getenv("MYSQL_PORT", "3306"))
MYSQL_DB = os.getenv("MYSQL_DB", "botnegar_safeganjineh")
MYSQL_USER = os.getenv("MYSQL_USER", "botnegar_ganjinehadmin")
MYSQL_PASSWORD = os.getenv("MYSQL_PASSWORD", "Zaqwer159")


def _connect() -> pymysql.connections.Connection:
    """
    Open a new MySQL connection.
    """
    return pymysql.connect(
        host=MYSQL_HOST,
        port=MYSQL_PORT,
        user=MYSQL_USER,
        password=MYSQL_PASSWORD,
        database=MYSQL_DB,
        charset="utf8mb4",
        autocommit=False,
        cursorclass=pymysql.cursors.Cursor,
    )


def init_db() -> None:
    """
    Create all tables if they do not already exist.
    Schema is kept equivalent to the previous sqlite version.
    """
    ddl_guarantees = """
        CREATE TABLE IF NOT EXISTS guarantees (
            code       VARCHAR(64)  PRIMARY KEY,
            user_id    BIGINT,
            name       VARCHAR(255),
            mobile     VARCHAR(32),
            store      VARCHAR(64),
            photo      TEXT,
            issued_at  VARCHAR(32),
            expires_at VARCHAR(32)
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci
    """

    ddl_stores = """
        CREATE TABLE IF NOT EXISTS stores (
            code VARCHAR(64) PRIMARY KEY,
            name VARCHAR(255) NOT NULL
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci
    """

    ddl_catalogs = """
        CREATE TABLE IF NOT EXISTS catalogs (
            code  VARCHAR(64) PRIMARY KEY,
            title VARCHAR(255) NOT NULL
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci
    """

    ddl_catalog_files = """
        CREATE TABLE IF NOT EXISTS catalog_files (
            id           BIGINT AUTO_INCREMENT PRIMARY KEY,
            catalog_code VARCHAR(64) NOT NULL,
            file_id      TEXT NOT NULL,
            file_name    TEXT,
            CONSTRAINT fk_catalog_files_catalog
                FOREIGN KEY (catalog_code) REFERENCES catalogs(code)
                ON DELETE CASCADE
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci
    """

    idx_catalog_files_code = """
        CREATE INDEX idx_catalog_files_code
        ON catalog_files (catalog_code)
    """

    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(ddl_guarantees)
            cursor.execute(ddl_stores)
            cursor.execute(ddl_catalogs)
            cursor.execute(ddl_catalog_files)
            try:
                cursor.execute(idx_catalog_files_code)
            except Exception:
                # ignore if index already exists
                pass
        conn.commit()


# ------------- guarantees -------------

def add_guarantee(data: Dict[str, Any]) -> None:
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                """
                INSERT INTO guarantees
                    (code, user_id, name, mobile, store, photo, issued_at, expires_at)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s)
                """,
                (
                    data["code"],
                    data.get("user_id"),
                    data.get("name"),
                    data.get("mobile"),
                    data.get("store"),
                    data.get("photo"),
                    data.get("issued_at"),
                    data.get("expires_at"),
                ),
            )
        conn.commit()


def get_by_code(code: str):
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute("SELECT * FROM guarantees WHERE code = %s", (code,))
            return cursor.fetchone()


def search_guarantees(query: str) -> List[Tuple]:
    like = f"%{query}%"
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                """
                SELECT * FROM guarantees
                WHERE name   LIKE %s
                   OR mobile LIKE %s
                   OR store  LIKE %s
                   OR code   LIKE %s
                ORDER BY issued_at DESC
                """,
                (like, like, like, like),
            )
            return cursor.fetchall()


def get_by_user(user_id: int) -> List[Tuple]:
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                """
                SELECT * FROM guarantees
                WHERE user_id = %s
                ORDER BY issued_at DESC
                """,
                (user_id,),
            )
            return cursor.fetchall()


def list_all(limit: int = 50) -> List[Tuple]:
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                """
                SELECT * FROM guarantees
                ORDER BY issued_at DESC
                LIMIT %s
                """,
                (limit,),
            )
            return cursor.fetchall()


def export_all() -> pd.DataFrame:
    with _connect() as conn:
        return pd.read_sql_query("SELECT * FROM guarantees", conn)


# ------------- stores -------------

def list_stores() -> List[Tuple[str, str]]:
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute("SELECT code, name FROM stores ORDER BY name")
            return cursor.fetchall()


def get_store_by_code(code: str) -> Optional[Tuple[str, str]]:
    clean = (code or "").strip().upper()
    if not clean:
        return None
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                "SELECT code, name FROM stores WHERE code = %s",
                (clean,),
            )
            return cursor.fetchone()


def add_store(code: str, name: str) -> None:
    clean_code = (code or "").strip().upper()
    clean_name = (name or "").strip()
    if not clean_code:
        raise ValueError("Store code is required")
    if not clean_name:
        raise ValueError("Store name is required")
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                "INSERT INTO stores(code, name) VALUES (%s, %s)",
                (clean_code, clean_name),
            )
        conn.commit()


def update_store(
    code: str,
    new_name: Optional[str] = None,
    new_code: Optional[str] = None,
) -> int:
    current_code = (code or "").strip().upper()
    if not current_code:
        return 0

    updates: List[str] = []
    params: List[Any] = []

    if new_name is not None:
        updates.append("name = %s")
        params.append((new_name or "").strip())
    if new_code is not None:
        updates.append("code = %s")
        params.append((new_code or "").strip().upper())

    if not updates:
        return 0

    params.append(current_code)

    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                f"UPDATE stores SET {', '.join(updates)} WHERE code = %s",
                params,
            )
        conn.commit()
        return cursor.rowcount


def search_stores(query: str, limit: int = 20) -> List[Tuple[str, str]]:
    text = (query or "").strip().upper()
    if not text:
        return []
    pattern = f"%{text}%"
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                """
                SELECT code, name FROM stores
                WHERE UPPER(code) LIKE %s OR UPPER(name) LIKE %s
                ORDER BY name
                LIMIT %s
                """,
                (pattern, pattern, limit),
            )
            return cursor.fetchall()


# ------------- catalogs -------------

def add_catalog(code: str, title: str) -> None:
    clean_code = (code or "").strip().upper()
    clean_title = (title or "").strip()
    if not clean_code:
        raise ValueError("Catalog code is required")
    if not clean_title:
        raise ValueError("Catalog title is required")
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                "INSERT INTO catalogs(code, title) VALUES (%s, %s)",
                (clean_code, clean_title),
            )
        conn.commit()


def update_catalog(
    code: str,
    new_title: Optional[str] = None,
    new_code: Optional[str] = None,
) -> int:
    current_code = (code or "").strip().upper()
    if not current_code:
        return 0

    updates: List[str] = []
    params: List[Any] = []

    if new_title is not None:
        updates.append("title = %s")
        params.append((new_title or "").strip())
    if new_code is not None:
        updates.append("code = %s")
        params.append((new_code or "").strip().upper())

    if not updates:
        return 0

    params.append(current_code)

    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                f"UPDATE catalogs SET {', '.join(updates)} WHERE code = %s",
                params,
            )
            if new_code is not None:
                new_code_clean = (new_code or "").strip().upper()
                cursor.execute(
                    "UPDATE catalog_files SET catalog_code = %s WHERE catalog_code = %s",
                    (new_code_clean, current_code),
                )
        conn.commit()
        return cursor.rowcount


def delete_catalog(code: str) -> int:
    clean_code = (code or "").strip().upper()
    if not clean_code:
        return 0
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute("DELETE FROM catalogs WHERE code = %s", (clean_code,))
        conn.commit()
        return cursor.rowcount


def list_catalogs() -> List[Tuple[str, str, int]]:
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                """
                SELECT c.code, c.title, COUNT(f.id) AS file_count
                FROM catalogs c
                LEFT JOIN catalog_files f ON f.catalog_code = c.code
                GROUP BY c.code, c.title
                ORDER BY c.title
                """
            )
            return cursor.fetchall()


def get_catalog_by_code(code: str) -> Optional[Tuple[str, str]]:
    clean_code = (code or "").strip().upper()
    if not clean_code:
        return None
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                "SELECT code, title FROM catalogs WHERE code = %s",
                (clean_code,),
            )
            return cursor.fetchone()


def search_catalogs(query: str, limit: int = 20) -> List[Tuple[str, str, int]]:
    text = (query or "").strip().upper()
    if not text:
        return []
    pattern = f"%{text}%"
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                """
                SELECT c.code, c.title, COUNT(f.id) AS file_count
                FROM catalogs c
                LEFT JOIN catalog_files f ON f.catalog_code = c.code
                WHERE UPPER(c.code) LIKE %s OR UPPER(c.title) LIKE %s
                GROUP BY c.code, c.title
                ORDER BY c.title
                LIMIT %s
                """,
                (pattern, pattern, limit),
            )
            return cursor.fetchall()


def replace_catalog_files(code: str, files: List[Tuple[str, Optional[str]]]) -> None:
    clean_code = (code or "").strip().upper()
    if not clean_code:
        raise ValueError("Catalog code is required")
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                "DELETE FROM catalog_files WHERE catalog_code = %s",
                (clean_code,),
            )
            if files:
                rows = [(clean_code, fid, fname) for fid, fname in files]
                cursor.executemany(
                    "INSERT INTO catalog_files(catalog_code, file_id, file_name) VALUES (%s, %s, %s)",
                    rows,
                )
        conn.commit()


def append_catalog_files(code: str, files: List[Tuple[str, Optional[str]]]) -> None:
    clean_code = (code or "").strip().upper()
    if not clean_code:
        raise ValueError("Catalog code is required")
    if not files:
        return
    with _connect() as conn:
        with conn.cursor() as cursor:
            rows = [(clean_code, fid, fname) for fid, fname in files]
            cursor.executemany(
                "INSERT INTO catalog_files(catalog_code, file_id, file_name) VALUES (%s, %s, %s)",
                rows,
            )
        conn.commit()


def list_catalog_files(code: str) -> List[Tuple[str, Optional[str]]]:
    clean_code = (code or "").strip().upper()
    if not clean_code:
        return []
    with _connect() as conn:
        with conn.cursor() as cursor:
            cursor.execute(
                "SELECT file_id, file_name FROM catalog_files WHERE catalog_code = %s ORDER BY id ASC",
                (clean_code,),
            )
            return cursor.fetchall()
