from __future__ import annotations

import logging
import math
import re
import unicodedata
from io import BytesIO
from typing import Any, Dict, List, Optional

from pypdf import PdfReader
from telegram import (
    InlineKeyboardButton,
    InlineKeyboardMarkup,
    ReplyKeyboardMarkup,
    Update,
)
from telegram.ext import (
    Application,
    CallbackQueryHandler,
    CommandHandler,
    ContextTypes,
    MessageHandler,
    filters,
)

from .config import load_settings
from .database import Database
from .staff import (
    ensure_owner,
    is_owner_or_admin,
    register_staff_handlers,
    show_staff_paged,
    staff_on_callback,
)

try:
    from pptx import Presentation  # type: ignore
except Exception:  # pragma: no cover
    Presentation = None

logger = logging.getLogger("negarprint.bot")
logging.basicConfig(
    level=logging.INFO, format="%(asctime)s [%(levelname)s] %(name)s: %(message)s"
)

# --------------------------------------------------------------------------- labels

L_START_ORDER = "ثبت درخواست"
L_TRACK = "پیگیری درخواست"
L_PRICES = "تعرفه‌ها"

L_ADMIN_PANEL = "مدیریت (ادمین) 🔧"
L_OPERATOR_PANEL = "پنل اپراتور 🛠"

L_BACK = "↩️ بازگشت"
L_CANCEL = "⛔️ انصراف / منوی اصلی"

ADMIN_CAT_TEAM = "تیم 👥"
ADMIN_CAT_ORDERS = "سفارشات 📦"
ADMIN_CAT_REPORTS = "گزارشات مالی 📊"
ADMIN_CAT_PRICES = "قیمت‌ها 💵"

OP_ORDERS = "سفارشات"
OP_TOGGLE_DUTY = "تغییر وضعیت من"

OP_GALLERY = "گالری سفارش‌ها"
OP_SEARCH = "جستجوی سفارش"

L_ALL = "تمام صفحات"
L_EVEN = "صفحات زوج"
L_ODD = "صفحات فرد"
L_RANGE = "بازه انتخابی"

L_SINGLE = "تک‌رو"
L_DOUBLE = "پشت‌ورو"

L_PAGES_OK = "✔️ تایید صفحات"
L_PAGES_REDO = "🔁 محاسبه دوباره"
L_PAGES_EDIT = "✍️ ورود دستی"

L_PAY_ONLINE = "پرداخت آنلاین"
L_PAY_CASH = "پرداخت حضوری"

PAPER_SIZES = ["A4", "A5", "A3"]
PAPER_TYPES_MAP = {"plain": "معمولی 80گرم", "glossy": "گلاسه 130گرم"}
COLOUR_MAP = {"bw": "سیاه‌وسفید", "color": "رنگی"}
FILE_KIND_MAP = {"print": "پرینت", "copy": "کپی"}

WORD_MIMES = {
    "application/msword",
    "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
}
PPT_MIMES = {
    "application/vnd.ms-powerpoint",
    "application/vnd.openxmlformats-officedocument.presentationml.presentation",
}
ALLOWED_DOC_MIMES = {
    "application/pdf",
    *WORD_MIMES,
    *PPT_MIMES,
}
NUP_OPTIONS = ["1", "2", "4", "6", "8"]

# --------------------------------------------------------------------------- normalizers / intents

_FA_DIGITS = str.maketrans("۰۱۲۳۴۵۶۷۸۹٠١٢٣٤٥٦٧٨٩", "01234567890123456789")
_CTRL = {0x200C, 0x200D, 0x200E, 0x200F}
_RX_NONWORDS = re.compile(r"[^\w\s؀-ۿ]")


def _norm(s: str) -> str:
    if not s:
        return ""
    s = unicodedata.normalize("NFKC", s).translate(_FA_DIGITS)
    s = "".join(ch for ch in s if ord(ch) not in _CTRL)
    return s.strip()


def _clean(s: str) -> str:
    return _RX_NONWORDS.sub("", _norm(s))


def _eq(a: str, b: str) -> bool:
    return _norm(a) == _norm(b)


def wants_show_prices(s: str) -> bool:
    b = _clean(s)
    return (("تعرفه" in b) or ("قیمت" in b)) and any(
        x in b for x in ("نمایش", "مشاهده", "لیست")
    )


def wants_edit_prices(s: str) -> bool:
    b = _clean(s)
    return ("تعرفه" in b) and any(
        x in b for x in ("ثبت", "ویرایش", "ثبتویرایش", "افزودن")
    )


def _rev(d: Dict[str, str]) -> Dict[str, str]:
    return {v: k for k, v in d.items()}


# --------------------------------------------------------------------------- keyboards


def rkm(rows: List[List[str]]) -> ReplyKeyboardMarkup:
    nav_row = [L_BACK, L_CANCEL]
    return ReplyKeyboardMarkup(rows + [nav_row], resize_keyboard=True)


def main_menu_for(
    user_id: int, context: ContextTypes.DEFAULT_TYPE
) -> ReplyKeyboardMarkup:
    db: Database = context.application.bot_data["db"]  # type: ignore[assignment]
    is_admin = False
    is_operator = False

    try:
        is_admin = db.is_admin(user_id)
    except Exception:
        is_admin = False

    if not is_admin:
        try:
            for r in db.list_staff(role="operator"):
                if int(r.get("user_id") or 0) == int(user_id) and int(
                    r.get("active", 1) or 1
                ):
                    is_operator = True
                    break
        except Exception:
            is_operator = False

    rows: List[List[str]] = []
    if is_admin:
        rows.append([L_ADMIN_PANEL])
    elif is_operator:
        rows.append([L_OPERATOR_PANEL])
    else:
        rows = [
            [L_START_ORDER, L_TRACK],
            [L_PRICES],
        ]
    return ReplyKeyboardMarkup(rows, resize_keyboard=True)


def admin_root_menu_kb() -> ReplyKeyboardMarkup:
    return rkm(
        [[ADMIN_CAT_TEAM, ADMIN_CAT_ORDERS], [ADMIN_CAT_REPORTS, ADMIN_CAT_PRICES]]
    )


def admin_orders_menu_kb() -> ReplyKeyboardMarkup:
    return rkm(
        [
            ["📦 همه سفارش‌ها", "🟢 فقط بازها"],
            ["✅ فقط انجام‌شده", "🔍 جستجوی سفارش"],
        ]
    )


def team_menu_kb() -> ReplyKeyboardMarkup:
    return rkm([["👥 لیست ادمین‌ها", "👨‍🔧 لیست اپراتورها"]])


def prices_menu_kb() -> ReplyKeyboardMarkup:
    return rkm([["📄 نمایش تعرفه‌ها", "⚙️ ثبت/ویرایش تعرفه"]])


def operator_menu_kb() -> ReplyKeyboardMarkup:
    return rkm([[OP_ORDERS, OP_TOGGLE_DUTY]])


def operator_orders_menu_kb() -> ReplyKeyboardMarkup:
    return rkm([[OP_GALLERY, OP_SEARCH]])


def page_range_kb() -> ReplyKeyboardMarkup:
    return rkm([[L_ALL, L_EVEN, L_ODD], [L_RANGE]])


def duplex_kb() -> ReplyKeyboardMarkup:
    return rkm([[L_SINGLE, L_DOUBLE]])


def copies_kb() -> ReplyKeyboardMarkup:
    return rkm([["1", "2", "3", "5", "10"]])


def paper_size_kb() -> ReplyKeyboardMarkup:
    return rkm([PAPER_SIZES])


def paper_type_kb() -> ReplyKeyboardMarkup:
    return rkm([list(PAPER_TYPES_MAP.values())])


def colour_kb() -> ReplyKeyboardMarkup:
    return rkm([list(COLOUR_MAP.values())])


def file_kind_kb() -> ReplyKeyboardMarkup:
    return rkm([list(FILE_KIND_MAP.values())])


def confirm_kb() -> ReplyKeyboardMarkup:
    return rkm([["➕ افزودن فایل دیگر", "✅ ثبت سفارش"]])


def pages_confirm_kb() -> ReplyKeyboardMarkup:
    return rkm([[L_PAGES_OK, L_PAGES_REDO, L_PAGES_EDIT]])


def pay_method_kb() -> ReplyKeyboardMarkup:
    return rkm([[L_PAY_ONLINE, L_PAY_CASH]])


def nup_kb() -> ReplyKeyboardMarkup:
    return rkm([NUP_OPTIONS])


# --------------------------------------------------------------------------- utils


async def count_pdf_pages(
    context: ContextTypes.DEFAULT_TYPE,
    file_id: str,
    approx_size: Optional[int] = None,
) -> Optional[int]:
    try:
        settings = context.application.bot_data.get("settings")
        base_to = getattr(settings, "timeout_pageops", 6) or 6
        big_mb = getattr(settings, "big_pdf_mb", 100) or 100
        read_to = base_to
        if approx_size and approx_size >= int(big_mb) * 1024 * 1024:
            read_to = max(base_to * 10, base_to + 60)

        f = await context.bot.get_file(file_id)
        bio = BytesIO()
        await f.download_to_memory(out=bio, read_timeout=read_to)
        bio.seek(0)
        return len(PdfReader(bio).pages)
    except Exception:
        return None


async def count_office_pages(
    context: ContextTypes.DEFAULT_TYPE,
    file_id: str,
    mime: str,
    approx_size: Optional[int] = None,
) -> Optional[int]:
    if mime in PPT_MIMES and Presentation is not None:
        try:
            settings = context.application.bot_data.get("settings")
            base_to = getattr(settings, "timeout_pageops", 6) or 6
            big_mb = getattr(settings, "big_pdf_mb", 100) or 100
            read_to = base_to
            if approx_size and approx_size >= int(big_mb) * 1024 * 1024:
                read_to = max(base_to * 10, base_to + 60)

            f = await context.bot.get_file(file_id)
            bio = BytesIO()
            await f.download_to_memory(out=bio, read_timeout=read_to)
            bio.seek(0)
            prs = Presentation(bio)
            return len(prs.slides)
        except Exception:
            return None
    return None


def parse_page_expr(expr: str, max_pages: int) -> List[int]:
    s = _norm(expr).replace(" ", "")
    if not s:
        raise ValueError("empty")
    pages: set[int] = set()
    for tok in s.split(","):
        if not re.fullmatch(r"\d+(-\d+)?", tok):
            raise ValueError(tok)
        if "-" in tok:
            a, b = map(int, tok.split("-", 1))
            if not (1 <= a <= b <= max_pages):
                raise ValueError(tok)
            pages.update(range(a, b + 1))
        else:
            x = int(tok)
            if not (1 <= x <= max_pages):
                raise ValueError(tok)
            pages.add(x)
    return sorted(pages)


def _estimate_order_price(db: Database, data: Dict[str, Any]) -> Optional[int]:
    files = data.get("files") or []
    if not isinstance(files, list):
        return None
    total = 0
    had_any_price = False
    for f in files:
        try:
            if f.get("selected_pages"):
                pages = len(f["selected_pages"])
            else:
                pages = int(f.get("page_count") or 0)
            if pages <= 0:
                continue
            duplex = f.get("duplex") or "single"
            copies = int(f.get("copies") or 1)
            paper_size = f.get("paper_size") or "A4"
            color = f.get("color") or "bw"
            paper_type = f.get("paper_type") or "plain"
            file_kind = f.get("file_kind") or "print"
            pages_per_sheet = int(f.get("pages_per_sheet") or 1)
            if pages_per_sheet <= 0:
                pages_per_sheet = 1
            sheets_per_copy = math.ceil(pages / pages_per_sheet)
            sheets = sheets_per_copy * copies
            tariff = db.get_tariff(
                file_kind=file_kind,
                paper_size=paper_size,
                color=color,
                duplex=duplex,
                paper_type=paper_type,
            )
            if not tariff:
                continue
            had_any_price = True
            price_per_sheet = int(tariff["price_per_sheet"])
            total += sheets * price_per_sheet
        except Exception:
            continue
    return total if had_any_price else None


def _build_order_text(
    data: Dict[str, Any],
    order_id: Optional[int] = None,
    *,
    total_price: Optional[int] = None,
) -> str:
    profile = data.get("profile") or {}
    files = data.get("files") or []
    name = profile.get("name", "")
    phone = profile.get("phone", "")
    uname = profile.get("username", "-")
    lines: List[str] = []

    lines.append("<b>🆕 سفارش جدید</b>" + (f" | <b>#{order_id}</b>" if order_id else ""))
    if name:
        lines.append(f"👤 نام: <b>{name}</b>")
    if uname:
        lines.append(f"🟦 یوزرنیم: <b>{uname}</b>")
    if uid := profile.get("uid", "-"):
        lines.append(f"🆔 آی‌دی: <code>{uid}</code>")
    if phone:
        lines.append(f"📞 تلفن: <b>{phone}</b>")
    lines.append(f"📎 تعداد فایل: <b>{len(files)}</b>")

    for i, o in enumerate(files, start=1):
        page_desc = o.get("page_range")
        if o.get("page_range") == "range" and o.get("selected_pages"):
            page_desc = f"{len(o['selected_pages'])} صفحه انتخابی"
        pt = o.get("paper_type", "")
        col = o.get("color", "")
        dup = "تک‌رو" if o.get("duplex") == "single" else "پشت‌ورو"
        copies = o.get("copies", 1)
        size = o.get("paper_size", "")
        pages_per_sheet = int(o.get("pages_per_sheet") or 1)
        lines += ["", f"🗂️ <b>فایل {i}</b>: {copies} نسخه | {size} | {pages_per_sheet} صفحه در هر برگ"]
        if page_desc:
            lines.append(f"• صفحات: {page_desc}")
        if pt:
            lines.append(f"• نوع کاغذ: {PAPER_TYPES_MAP.get(pt, pt)}")
        if col:
            lines.append(f"• نوع چاپ: {COLOUR_MAP.get(col, col)}")
        if dup:
            lines.append(f"• یک/دو رو: {dup}")

    if total_price is not None:
        lines.append("")
        lines.append(f"💰 مبلغ تقریبی: <b>{total_price:,} تومان</b>")

    return "\n".join(lines)


def _pages_summary(data: Dict[str, Any]) -> str:
    orders = data.get("files") or []
    lines = ["🔢 خلاصه صفحات:"]
    for i, f in enumerate(orders, 1):
        sel = f.get("selected_pages")
        if sel:
            lines.append(f"• فایل {i}: {len(sel)} صفحه (انتخابی)")
        else:
            lines.append(f"• فایل {i}: {f.get('page_count') or '?'} صفحه")
    return "\n".join(lines)


def _gallery_markup(
    oid: int, page: int, total: int, done: bool
) -> InlineKeyboardMarkup:
    prev_p = max(0, page - 1)
    next_p = min(total - 1, page + 1)
    btns = [
        InlineKeyboardButton("⬅️ قبلی", callback_data=f"GAL:NAV:{prev_p}"),
        InlineKeyboardButton("➡️ بعدی", callback_data=f"GAL:NAV:{next_p}"),
        InlineKeyboardButton("جزئیات", callback_data=f"ORDER_DETAIL:{oid}"),
    ]
    if done:
        btns.append(InlineKeyboardButton("✔️ انجام‌شده", callback_data="NOP"))
    else:
        btns.append(
            InlineKeyboardButton("✔️ انجام", callback_data=f"GAL:DONE:{oid}:{page}")
        )
    return InlineKeyboardMarkup([btns])


def _render_orders_list(rows: List[Dict[str, Any]]) -> str:
    if not rows:
        return "سفارشی یافت نشد."
    lines = ["لیست سفارش‌ها:"]
    for r in rows:
        lines.append(
            f"- #{r['id']} | {r['status']} | user={r['user_id']}"
        )
    return "\n".join(lines)


# --------------------------------------------------------------------------- states

S_NONE = "none"
S_USER = "user"

S_WAIT_FILES = "wait_files"
S_PAGE_COUNT = "page_count"
S_PAGE_RANGE = "page_range"
S_PAGE_RANGE_CUSTOM = "page_range_custom"
S_PAGECOUNT_CONFIRM = "pagecount_confirm"
S_NUP = "nup"
S_DUPLEX = "duplex"
S_COPIES = "copies"
S_PAPER_SIZE = "paper_size"
S_PAPER_TYPE = "paper_type"
S_COLOR = "color"
S_CONFIRM = "confirm"
S_PAY_METHOD = "pay_method"

S_MG_CAT = "mg_cat"

S_TAR_FILE_KIND = "tar_file_kind"
S_TAR_SIZE = "tar_size"
S_TAR_COLOR = "tar_color"
S_TAR_DUPLEX = "tar_duplex"
S_TAR_PTYPE = "tar_ptype"
S_TAR_PRICE = "tar_price"

S_OP_MENU = "op_menu"
S_OP_ORDERS = "op_orders"
S_OP_GALLERY = "op_gallery"
S_OP_SEARCH = "op_search"
S_OP_SEARCH_QUERY = "op_search_query"


def set_state(ctx: ContextTypes.DEFAULT_TYPE, st: str) -> None:
    ctx.user_data["state"] = st


# --------------------------------------------------------------------------- main bot


def run_bot() -> None:
    settings = load_settings()
    if not settings.telegram_bot_token:
        raise RuntimeError("TELEGRAM_BOT_TOKEN missing in configuration")
    db = Database(settings.database_path)
    logger.info("DB at %s", settings.database_path)

    app = Application.builder().token(settings.telegram_bot_token).build()
    app.bot_data["db"] = db
    app.bot_data["settings"] = settings

    async def on_error(update: object, context: ContextTypes.DEFAULT_TYPE) -> None:  # type: ignore[override]
        logger.exception("Unhandled error: %s", context.error)

    app.add_error_handler(on_error)

    ensure_owner(db, settings)
    register_staff_handlers(app, db, settings)

    # -------------------------------------------------------------- navigation helpers

    async def goto_home(
        update: Update, context: ContextTypes.DEFAULT_TYPE, msg: Optional[str] = None
    ):
        context.user_data.clear()
        uid = update.effective_user.id
        await update.effective_message.reply_text(
            msg or "منوی اصلی:",
            reply_markup=main_menu_for(uid, context),
        )
        set_state(context, S_NONE)

    async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
        await goto_home(update, context, "ربات فعال است ✅")

    # -------------------------------------------------------------- customer order flow

    async def begin_order(update: Update, context: ContextTypes.DEFAULT_TYPE):
        d = context.user_data.setdefault("order", {"files": []})
        d["profile"] = {
            "uid": update.effective_user.id,
            "username": ("@" + (update.effective_user.username or ""))
            if update.effective_user.username
            else "-",
            "name": update.effective_user.full_name,
            "phone": context.user_data.get("phone", ""),
        }
        set_state(context, S_WAIT_FILES)
        await update.effective_message.reply_text(
            "فایل‌های خود را ارسال کنید (PDF، Word، PowerPoint یا عکس). سپس «✅ ثبت سفارش» را بزنید.",
            reply_markup=confirm_kb(),
        )

    async def on_document(update: Update, context: ContextTypes.DEFAULT_TYPE):
        if context.user_data.get("state") != S_WAIT_FILES:
            return

        doc = update.effective_message.document
        if not doc:
            return

        mime = (doc.mime_type or "").lower()
        if mime not in ALLOWED_DOC_MIMES:
            return await update.effective_message.reply_text(
                "این نوع فایل پشتیبانی نمی‌شود.\n"
                "لطفاً فقط PDF، Word، PowerPoint یا عکس ارسال کنید.",
                reply_markup=confirm_kb(),
            )

        pages: Optional[int] = None
        if mime == "application/pdf":
            pages = await count_pdf_pages(
                context,
                doc.file_id,
                approx_size=(doc.file_size or None),
            )
        else:
            pages = await count_office_pages(
                context,
                doc.file_id,
                mime,
                approx_size=(doc.file_size or None),
            )

        context.user_data.setdefault("order", {"files": []})
        context.user_data["order"]["files"].append(
            {
                "type": "document",
                "file_id": doc.file_id,
                "message_id": update.effective_message.message_id,
                "page_count": pages,
                "page_range": "all",
                "duplex": "single",
                "copies": 1,
                "paper_size": "A4",
                "paper_type": "plain",
                "color": "bw",
                "file_kind": "print",
                "pages_per_sheet": 1,
            }
        )

        await update.effective_message.reply_text(
            f"فایل دریافت شد. تعداد صفحات: {pages if pages else 'نامشخص'}",
            reply_markup=confirm_kb(),
        )

    async def on_photo(update: Update, context: ContextTypes.DEFAULT_TYPE):
        if context.user_data.get("state") != S_WAIT_FILES:
            return
        ph = (update.effective_message.photo or [])[-1]
        if not ph:
            return
        context.user_data.setdefault("order", {"files": []})
        context.user_data["order"]["files"].append(
            {
                "type": "photo",
                "file_id": ph.file_id,
                "message_id": update.effective_message.message_id,
                "page_count": 1,
                "page_range": "all",
                "duplex": "single",
                "copies": 1,
                "paper_size": "A4",
                "paper_type": "plain",
                "color": "bw",
                "file_kind": "print",
                "pages_per_sheet": 1,
            }
        )
        await update.effective_message.reply_text(
            "عکس دریافت شد.", reply_markup=confirm_kb()
        )

    async def _ask_pagecount_confirm(
        update: Update, context: ContextTypes.DEFAULT_TYPE
    ):
        set_state(context, S_PAGECOUNT_CONFIRM)
        od = context.user_data.get("order") or {}
        body = _pages_summary(od) + "\n\nتعداد صفحات را تایید می‌کنید؟"
        await update.effective_message.reply_text(body, reply_markup=pages_confirm_kb())

    async def confirm_or_continue(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = _norm(update.effective_message.text or "")
        if _eq(txt, "➕ افزودن فایل دیگر"):
            return
        if not _eq(txt, "✅ ثبت سفارش"):
            return
        files = context.user_data.get("order", {}).get("files", [])
        if not files:
            return await update.effective_message.reply_text(
                "هیچ فایلی ثبت نشده است.", reply_markup=confirm_kb()
            )
        need = [i for i, f in enumerate(files) if f.get("page_count") in (None, 0)]
        if need:
            context.user_data["need_pages_idx"] = need
            set_state(context, S_PAGE_COUNT)
            return await update.effective_message.reply_text(
                "تعداد صفحات را برای فایل‌های نامشخص، به‌ترتیب وارد کنید.",
                reply_markup=rkm([[]]),
            )
        set_state(context, S_PAGE_RANGE)
        return await update.effective_message.reply_text(
            "صفحات مورد نظر:", reply_markup=page_range_kb()
        )

    async def handle_page_count(update: Update, context: ContextTypes.DEFAULT_TYPE):
        try:
            n = int(_norm(update.effective_message.text or ""))
            assert n > 0
        except Exception:
            return await update.effective_message.reply_text(
                "عدد معتبر وارد کنید.", reply_markup=rkm([[]])
            )
        idxs = context.user_data.get("need_pages_idx", [])
        if not idxs:
            return await goto_home(update, context, "اشکال در جریان سفارش.")
        files = context.user_data["order"]["files"]
        i = idxs.pop(0)
        files[i]["page_count"] = n
        if idxs:
            context.user_data["need_pages_idx"] = idxs
            return await update.effective_message.reply_text(
                "فایل بعدی:", reply_markup=rkm([[]])
            )
        from_edit = bool(context.user_data.pop("pagecount_reedit", False))
        if from_edit:
            return await _ask_pagecount_confirm(update, context)
        set_state(context, S_PAGE_RANGE)
        return await update.effective_message.reply_text(
            "صفحات مورد نظر:", reply_markup=page_range_kb()
        )

    async def handle_page_range(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        if not any(_eq(txt, x) for x in (L_ALL, L_EVEN, L_ODD, L_RANGE)):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=page_range_kb()
            )
        files = context.user_data["order"]["files"]
        mapping = {
            L_ALL: "all",
            L_EVEN: "even",
            L_ODD: "odd",
            L_RANGE: "range",
        }
        choice = mapping[[k for k in mapping if _eq(txt, k)][0]]
        for f in files:
            f["page_range"] = choice
        if choice == "range":
            set_state(context, S_PAGE_RANGE_CUSTOM)
            mx = max(f.get("page_count") or 1 for f in files)
            return await update.effective_message.reply_text(
                f"عبارت صفحات (۱-۳,۵) تا {mx}:",
                reply_markup=rkm([[]]),
            )
        set_state(context, S_NUP)
        return await update.effective_message.reply_text(
            "چند صفحه در هر برگ چاپ شود؟", reply_markup=nup_kb()
        )

    async def handle_page_range_custom(
        update: Update, context: ContextTypes.DEFAULT_TYPE
    ):
        txt = update.effective_message.text or ""
        mx = max(f.get("page_count") or 1 for f in context.user_data["order"]["files"])
        try:
            pages = parse_page_expr(txt, mx)
        except Exception:
            return await update.effective_message.reply_text(
                "عبارت نامعتبر است.", reply_markup=rkm([[]])
            )
        for f in context.user_data["order"]["files"]:
            f["selected_pages"] = pages

        set_state(context, S_NUP)
        return await update.effective_message.reply_text(
            "چند صفحه در هر برگ چاپ شود؟", reply_markup=nup_kb()
        )

    async def handle_nup(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = _norm(update.effective_message.text or "")
        try:
            n = int(txt)
            assert n >= 1
        except Exception:
            return await update.effective_message.reply_text(
                "یک عدد معتبر وارد کنید (مثلاً 1، 2، 4).",
                reply_markup=nup_kb(),
            )

        files = context.user_data.get("order", {}).get("files", [])
        for f in files:
            f["pages_per_sheet"] = n

        set_state(context, S_DUPLEX)
        return await update.effective_message.reply_text(
            "یک‌رو/دو‌رو:",
            reply_markup=duplex_kb(),
        )

    async def handle_duplex(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        mapping = {L_SINGLE: "single", L_DOUBLE: "double"}
        if not any(_eq(txt, k) for k in mapping):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=duplex_kb()
            )
        value = mapping[[k for k in mapping if _eq(txt, k)][0]]
        for f in context.user_data["order"]["files"]:
            f["duplex"] = value
        set_state(context, S_COPIES)
        return await update.effective_message.reply_text(
            "تعداد نسخه:", reply_markup=copies_kb()
        )

    async def handle_copies(update: Update, context: ContextTypes.DEFAULT_TYPE):
        try:
            n = int(_norm(update.effective_message.text or ""))
            assert 1 <= n <= 100
        except Exception:
            return await update.effective_message.reply_text(
                "عدد بین ۱ تا ۱۰۰.", reply_markup=copies_kb()
            )
        for f in context.user_data["order"]["files"]:
            f["copies"] = n
        set_state(context, S_PAPER_SIZE)
        return await update.effective_message.reply_text(
            "اندازه کاغذ:", reply_markup=paper_size_kb()
        )

    async def handle_paper_size(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        if not any(_eq(txt, x) for x in PAPER_SIZES):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=paper_size_kb()
            )
        choice = [x for x in PAPER_SIZES if _eq(txt, x)][0]
        for f in context.user_data["order"]["files"]:
            f["paper_size"] = choice
        set_state(context, S_PAPER_TYPE)
        return await update.effective_message.reply_text(
            "نوع کاغذ:", reply_markup=paper_type_kb()
        )

    async def handle_paper_type(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        rev = _rev(PAPER_TYPES_MAP)
        if not any(_eq(txt, x) for x in rev):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=paper_type_kb()
            )
        choice = rev[[x for x in rev if _eq(txt, x)][0]]
        for f in context.user_data["order"]["files"]:
            f["paper_type"] = choice
        set_state(context, S_COLOR)
        return await update.effective_message.reply_text(
            "نوع چاپ:", reply_markup=colour_kb()
        )

    async def handle_color(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        rev = _rev(COLOUR_MAP)
        if not any(_eq(txt, x) for x in rev):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=colour_kb()
            )
        choice = rev[[x for x in rev if _eq(txt, x)][0]]
        for f in context.user_data["order"]["files"]:
            f["color"] = choice
        return await _ask_pagecount_confirm(update, context)

    async def handle_pagecount_confirm(
        update: Update, context: ContextTypes.DEFAULT_TYPE
    ):
        txt = update.effective_message.text or ""
        if _eq(txt, L_PAGES_OK):
            set_state(context, S_CONFIRM)
            return await update.effective_message.reply_text(
                "بررسی و تایید اطلاعات سفارش:",
                reply_markup=confirm_kb(),
            )
        if _eq(txt, L_PAGES_REDO):
            files = context.user_data.get("order", {}).get("files", [])
            changed = False
            for f in files:
                if f.get("type") == "document" and not f.get("page_count"):
                    pc = await count_pdf_pages(context, f["file_id"])
                    if pc:
                        f["page_count"] = pc
                        changed = True
            msg = _pages_summary(context.user_data.get("order", {}))
            if not changed:
                msg += "\n(تغییری نداشت.)"
            return await update.effective_message.reply_text(
                msg, reply_markup=pages_confirm_kb()
            )
        if _eq(txt, L_PAGES_EDIT):
            files = context.user_data.get("order", {}).get("files", [])
            need_pages = [i for i, _ in enumerate(files)]
            context.user_data["need_pages_idx"] = need_pages
            context.user_data["pagecount_reedit"] = True
            set_state(context, S_PAGE_COUNT)
            return await update.effective_message.reply_text(
                "تعداد صفحات را به‌ترتیب برای هر فایل وارد کنید.",
                reply_markup=rkm([[]]),
            )
        return await update.effective_message.reply_text(
            "از دکمه‌ها استفاده کنید.", reply_markup=pages_confirm_kb()
        )

    async def _finalize_and_forward(update: Update, context: ContextTypes.DEFAULT_TYPE):
        od = context.user_data.get("order") or {}
        files = od.get("files") or []
        if not files:
            return await goto_home(update, context, "سفارشی ثبت نشده است.")

        total_price = None
        try:
            total_price = _estimate_order_price(db, od)
        except Exception:
            total_price = None

        oid = db.save_order(update.effective_user.id, "new", od)
        caption = _build_order_text(od, order_id=oid, total_price=total_price)

        admin_chat_id = getattr(settings, "admin_chat_id", None)
        if admin_chat_id:
            btn = InlineKeyboardMarkup(
                [
                    [
                        InlineKeyboardButton(
                            f"جزئیات #{oid}", callback_data=f"ORDER_DETAIL:{oid}"
                        )
                    ]
                ]
            )
            try:
                await context.bot.send_message(
                    chat_id=admin_chat_id,
                    text=caption,
                    parse_mode="HTML",
                    reply_markup=btn,
                )
            except Exception:
                pass

        target: Optional[int] = None
        try:
            target = db.choose_operator(fallback_admin=False)
        except Exception:
            target = None

        if not target:
            op_default = getattr(settings, "operator_chat_id", None)
            if not op_default:
                op_list = getattr(settings, "operator_chat_ids", None)
                if isinstance(op_list, (list, tuple)) and op_list:
                    op_default = op_list[0]
            if op_default:
                try:
                    target = int(op_default)
                except Exception:
                    target = op_default  # type: ignore[assignment]

        if not target:
            target = admin_chat_id

        if not target:
            return await goto_home(
                update, context, f"سفارش #{oid} ثبت شد ✅ (بدون مقصد برای فوروارد)"
            )

        for i, f in enumerate(files):
            try:
                await context.bot.copy_message(
                    chat_id=target,
                    from_chat_id=update.effective_chat.id,
                    message_id=f["message_id"],
                    caption=(caption if i == 0 else None),
                    parse_mode="HTML",
                )
            except Exception:
                continue

        await goto_home(update, context, f"سفارش #{oid} ثبت شد ✅")

    async def finalize_order(update: Update, context: ContextTypes.DEFAULT_TYPE):
        if context.user_data.get("state") != S_CONFIRM:
            return
        set_state(context, S_PAY_METHOD)
        await update.effective_message.reply_text(
            "روش پرداخت را انتخاب کنید:",
            reply_markup=pay_method_kb(),
        )

    async def handle_pay_method(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        if not any(_eq(txt, x) for x in (L_PAY_ONLINE, L_PAY_CASH)):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.",
                reply_markup=pay_method_kb(),
            )
        od = context.user_data.get("order") or {}
        od["payment_method"] = L_PAY_ONLINE if _eq(txt, L_PAY_ONLINE) else L_PAY_CASH
        await _finalize_and_forward(update, context)

    # -------------------------------------------------------------- operator

    async def op_root(update: Update, context: ContextTypes.DEFAULT_TYPE):
        set_state(context, S_OP_MENU)
        await update.effective_message.reply_text(
            "پنل اپراتور:", reply_markup=operator_menu_kb()
        )

    async def op_orders(update: Update, context: ContextTypes.DEFAULT_TYPE):
        set_state(context, S_OP_ORDERS)
        await update.effective_message.reply_text(
            "سفارشات:", reply_markup=operator_orders_menu_kb()
        )

    async def gallery_start(
        update: Update,
        context: ContextTypes.DEFAULT_TYPE,
        rows: Optional[List[Dict[str, Any]]] = None,
    ):
        if rows is None:
            rows = db.list_orders(statuses=["new", "assigned"], limit=200)
        if not rows:
            return await update.effective_message.reply_text(
                "سفارشی موجود نیست.",
                reply_markup=operator_orders_menu_kb(),
            )
        context.user_data["gallery_rows"] = rows
        context.user_data["gallery_page"] = 0
        await gallery_show(update, context, 0)

    async def gallery_show(
        update: Update, context: ContextTypes.DEFAULT_TYPE, page: int
    ):
        rows = context.user_data.get("gallery_rows", [])
        if not rows:
            return await update.effective_message.reply_text("موردی نیست.")
        total = len(rows)
        page = max(0, min(page, total - 1))
        context.user_data["gallery_page"] = page
        row = rows[page]
        done = row.get("status") == "done"
        total_price = _estimate_order_price(db, row.get("data") or {})
        cap = _build_order_text(
            row["data"], order_id=row["id"], total_price=total_price
        )
        kb = _gallery_markup(row["id"], page, total, done)
        try:
            await update.effective_message.reply_text(
                cap, parse_mode="HTML", reply_markup=kb
            )
        except Exception:
            await context.bot.send_message(
                chat_id=update.effective_chat.id,
                text=cap,
                parse_mode="HTML",
                reply_markup=kb,
            )

    async def on_gallery_nav(update: Update, context: ContextTypes.DEFAULT_TYPE):
        q = update.callback_query
        await q.answer()
        data = q.data or ""
        if data == "NOP":
            return
        if data.startswith("GAL:NAV:"):
            page = int(data.split(":")[2])
            rows = context.user_data.get("gallery_rows", [])
            if not rows:
                return
            total = len(rows)
            page = max(0, min(page, total - 1))
            row = rows[page]
            done = row.get("status") == "done"
            total_price = _estimate_order_price(db, row.get("data") or {})
            cap = _build_order_text(
                row["data"], order_id=row["id"], total_price=total_price
            )
            kb = _gallery_markup(row["id"], page, total, done)
            try:
                await q.edit_message_text(cap, parse_mode="HTML", reply_markup=kb)
            except Exception:
                await context.bot.send_message(
                    chat_id=q.message.chat.id,
                    text=cap,
                    parse_mode="HTML",
                    reply_markup=kb,
                )
        elif data.startswith("GAL:DONE:"):
            _, _, oid_str, page_str = data.split(":")
            oid = int(oid_str)
            page = int(page_str)
            db.set_order_status(oid, "done")
            rows = context.user_data.get("gallery_rows", [])
            for r in rows:
                if int(r["id"]) == oid:
                    r["status"] = "done"
            row = db.get_order(oid)
            if not row:
                return
            total_price = _estimate_order_price(db, row.get("data") or {})
            cap = _build_order_text(
                row["data"], order_id=oid, total_price=total_price
            )
            kb = _gallery_markup(oid, page, len(rows), True)
            try:
                await q.edit_message_text(cap, parse_mode="HTML", reply_markup=kb)
            except Exception:
                await context.bot.send_message(
                    chat_id=q.message.chat.id,
                    text=cap,
                    parse_mode="HTML",
                    reply_markup=kb,
                )

    async def op_search_start(update: Update, context: ContextTypes.DEFAULT_TYPE):
        set_state(context, S_OP_SEARCH_QUERY)
        await update.effective_message.reply_text(
            "آیدی سفارش یا موبایل را ارسال کنید:",
            reply_markup=rkm([[]]),
        )

    async def op_search_query(update: Update, context: ContextTypes.DEFAULT_TYPE):
        q = _norm(update.effective_message.text or "")
        rows = db.search_orders(q, limit=50)
        if not rows:
            return await update.effective_message.reply_text(
                "سفارشی یافت نشد.",
                reply_markup=operator_orders_menu_kb(),
            )
        await gallery_start(update, context, rows)

    # -------------------------------------------------------------- admin

    async def admin_root(update: Update, context: ContextTypes.DEFAULT_TYPE):
        set_state(context, S_MG_CAT)
        context.user_data["mg_cat"] = None
        await update.effective_message.reply_text(
            "مدیریت:", reply_markup=admin_root_menu_kb()
        )

    async def _admin_financial_report(update: Update, context: ContextTypes.DEFAULT_TYPE):
        rows = db.list_orders(limit=1000)
        total = len(rows)
        done_orders = [r for r in rows if r["status"] == "done"]
        open_orders = [r for r in rows if r["status"] != "done"]

        total_done = len(done_orders)
        total_open = len(open_orders)

        sum_all = 0
        sum_done = 0
        sum_open = 0

        for r in rows:
            val = _estimate_order_price(db, r.get("data") or {}) or 0
            sum_all += val
            if r["status"] == "done":
                sum_done += val
            else:
                sum_open += val

        body = [
            "📊 گزارش مالی کل:",
            f"- تعداد کل سفارش‌ها: <b>{total}</b>",
            f"  • انجام‌شده: <b>{total_done}</b>",
            f"  • درحال‌انتظار / باز: <b>{total_open}</b>",
            "",
            f"💰 مجموع تقریبی مبالغ همه سفارش‌ها: <b>{sum_all:,} تومان</b>",
            f"✅ مجموع مبالغ سفارش‌های انجام‌شده: <b>{sum_done:,} تومان</b>",
            f"🕓 مجموع مبالغ سفارش‌های باز: <b>{sum_open:,} تومان</b>",
        ]
        await update.effective_message.reply_text(
            "\n".join(body), parse_mode="HTML", reply_markup=admin_root_menu_kb()
        )

    async def admin_handle(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        ntx = _norm(txt)
        cat = context.user_data.get("mg_cat")
        if cat is None:
            if _eq(ntx, ADMIN_CAT_TEAM):
                context.user_data["mg_cat"] = "team"
                return await update.effective_message.reply_text(
                    "تیم:", reply_markup=team_menu_kb()
                )
            if _eq(ntx, ADMIN_CAT_ORDERS):
                context.user_data["mg_cat"] = "orders"
                return await update.effective_message.reply_text(
                    "مدیریت سفارشات:", reply_markup=admin_orders_menu_kb()
                )
            if _eq(ntx, ADMIN_CAT_REPORTS):
                context.user_data["mg_cat"] = "reports"
                return await _admin_financial_report(update, context)
            if _eq(ntx, ADMIN_CAT_PRICES):
                context.user_data["mg_cat"] = "prices"
                return await update.effective_message.reply_text(
                    "قیمت‌ها:", reply_markup=prices_menu_kb()
                )
            return await update.effective_message.reply_text(
                "یک دسته را انتخاب کنید.", reply_markup=admin_root_menu_kb()
            )

        if cat == "team":
            if _eq(ntx, "👥 لیست ادمین‌ها"):
                return await show_staff_paged(update, context, "admin", 0)
            if _eq(ntx, "👨‍🔧 لیست اپراتورها"):
                return await show_staff_paged(update, context, "operator", 0)
            return await update.effective_message.reply_text(
                "از دکمه‌های تیم استفاده کنید.", reply_markup=team_menu_kb()
            )

        if cat == "orders":
            if _eq(ntx, "📦 همه سفارش‌ها"):
                rows = db.list_orders(limit=50)
                body = _render_orders_list(rows)
                return await update.effective_message.reply_text(
                    body, reply_markup=admin_orders_menu_kb()
                )
            if _eq(ntx, "🟢 فقط بازها"):
                rows = db.list_orders(statuses=["new", "assigned"], limit=50)
                body = _render_orders_list(rows)
                return await update.effective_message.reply_text(
                    body, reply_markup=admin_orders_menu_kb()
                )
            if _eq(ntx, "✅ فقط انجام‌شده"):
                rows = db.list_orders(statuses=["done"], limit=50)
                body = _render_orders_list(rows)
                return await update.effective_message.reply_text(
                    body, reply_markup=admin_orders_menu_kb()
                )
            if _eq(ntx, "🔍 جستجوی سفارش"):
                set_state(context, S_OP_SEARCH_QUERY)
                return await op_search_start(update, context)
            return await update.effective_message.reply_text(
                "از منوی سفارشات استفاده کنید.", reply_markup=admin_orders_menu_kb()
            )

        if cat == "prices":
            if wants_show_prices(ntx) or _eq(ntx, "📄 نمایش تعرفه‌ها"):
                rows = db.list_tariffs()
                if not rows:
                    return await update.effective_message.reply_text(
                        "تعرفه‌ای ثبت نشده.", reply_markup=prices_menu_kb()
                    )
                body = "\n".join(
                    [
                        f"- {FILE_KIND_MAP.get(r['file_kind'], r['file_kind'])} | "
                        f"{r['paper_size']} | {COLOUR_MAP.get(r['color'], r['color'])} | "
                        f"{r['duplex']} | {PAPER_TYPES_MAP.get(r['paper_type'], r['paper_type'])} : "
                        f"{r['price_per_sheet']:,}"
                        for r in rows
                    ]
                )
                return await update.effective_message.reply_text(
                    body, reply_markup=prices_menu_kb()
                )
            if wants_edit_prices(ntx) or _eq(ntx, "⚙️ ثبت/ویرایش تعرفه"):
                context.user_data["tariff"] = {}
                set_state(context, S_TAR_FILE_KIND)
                return await update.effective_message.reply_text(
                    "نوع کار:", reply_markup=file_kind_kb()
                )
            return await update.effective_message.reply_text(
                "قیمت‌ها:", reply_markup=prices_menu_kb()
            )

        if cat == "reports":
            return await _admin_financial_report(update, context)

        return await update.effective_message.reply_text(
            "به منوی مدیریت بازگردید.", reply_markup=admin_root_menu_kb()
        )

    # -------------------------------------------------------------- tariff wizard

    async def tar_file_kind(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        rev = _rev(FILE_KIND_MAP)
        if not any(_eq(txt, x) for x in rev):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=file_kind_kb()
            )
        t = context.user_data.setdefault("tariff", {})
        t["file_kind"] = rev[[x for x in rev if _eq(txt, x)][0]]
        set_state(context, S_TAR_SIZE)
        return await update.effective_message.reply_text(
            "اندازه:", reply_markup=paper_size_kb()
        )

    async def tar_size(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        if not any(_eq(txt, x) for x in PAPER_SIZES):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=paper_size_kb()
            )
        t = context.user_data.setdefault("tariff", {})
        t["paper_size"] = [x for x in PAPER_SIZES if _eq(txt, x)][0]
        set_state(context, S_TAR_COLOR)
        return await update.effective_message.reply_text(
            "نوع چاپ:", reply_markup=colour_kb()
        )

    async def tar_color(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        rev = _rev(COLOUR_MAP)
        if not any(_eq(txt, x) for x in rev):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=colour_kb()
            )
        t = context.user_data.setdefault("tariff", {})
        t["color"] = rev[[x for x in rev if _eq(txt, x)][0]]
        set_state(context, S_TAR_DUPLEX)
        return await update.effective_message.reply_text(
            "یک‌رو/دو‌رو:", reply_markup=duplex_kb()
        )

    async def tar_duplex(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        mapping = {L_SINGLE: "single", L_DOUBLE: "double"}
        if not any(_eq(txt, k) for k in mapping):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=duplex_kb()
            )
        t = context.user_data.setdefault("tariff", {})
        t["duplex"] = mapping[[k for k in mapping if _eq(txt, k)][0]]
        set_state(context, S_TAR_PTYPE)
        return await update.effective_message.reply_text(
            "نوع کاغذ:", reply_markup=paper_type_kb()
        )

    async def tar_ptype(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        rev = _rev(PAPER_TYPES_MAP)
        if not any(_eq(txt, x) for x in rev):
            return await update.effective_message.reply_text(
                "از دکمه‌ها استفاده کنید.", reply_markup=paper_type_kb()
            )
        t = context.user_data.setdefault("tariff", {})
        t["paper_type"] = rev[[x for x in rev if _eq(txt, x)][0]]
        set_state(context, S_TAR_PRICE)
        return await update.effective_message.reply_text(
            "قیمت هر برگ (تومان):", reply_markup=rkm([[]])
        )

    async def tar_price(update: Update, context: ContextTypes.DEFAULT_TYPE):
        try:
            price = int(_norm(update.effective_message.text or ""))
            assert price >= 0
        except Exception:
            return await update.effective_message.reply_text(
                "عدد معتبر وارد کنید.", reply_markup=rkm([[]])
            )
        t = context.user_data.get("tariff", {})
        db.upsert_tariff(
            file_kind=t["file_kind"],
            paper_size=t["paper_size"],
            color=t["color"],
            duplex=t["duplex"],
            paper_type=t["paper_type"],
            price_per_sheet=price,
        )
        context.user_data.pop("tariff", None)
        set_state(context, S_MG_CAT)
        context.user_data["mg_cat"] = "prices"
        return await update.effective_message.reply_text(
            "تعرفه ذخیره شد ✔️", reply_markup=prices_menu_kb()
        )

    # -------------------------------------------------------------- callbacks

    async def on_cb(update: Update, context: ContextTypes.DEFAULT_TYPE):
        q = update.callback_query
        await q.answer()
        data = q.data or ""
        if data == "NOP":
            return
        if data.startswith("ORDER_DETAIL:"):
            try:
                oid = int(data.split(":", 1)[1])
            except Exception:
                return
            row = db.get_order(oid)
            if not row:
                return await q.edit_message_text("سفارش یافت نشد.")
            total_price = _estimate_order_price(db, row.get("data") or {})
            body = _build_order_text(
                row["data"], order_id=row["id"], total_price=total_price
            )
            try:
                await q.edit_message_text(body, parse_mode="HTML")
            except Exception:
                await context.bot.send_message(
                    chat_id=q.message.chat.id,
                    text=body,
                    parse_mode="HTML",
                )
            return
        if data.startswith("GAL:"):
            return await on_gallery_nav(update, context)
        if data.startswith("STAFF:"):
            return await staff_on_callback(update, context)

    # -------------------------------------------------------------- text router

    async def handle_text(update: Update, context: ContextTypes.DEFAULT_TYPE):
        txt = update.effective_message.text or ""
        ntx = _norm(txt)
        uid = update.effective_user.id
        st = context.user_data.get("state", S_NONE)

        # ------------------------ global navigation ------------------------
        if _eq(ntx, L_CANCEL):
            return await goto_home(update, context, "به منوی اصلی بازگشتید.")

        if _eq(ntx, L_BACK):
            # Order wizard
            if st == S_WAIT_FILES:
                return await goto_home(update, context, "ثبت سفارش لغو شد.")
            if st in {S_PAGE_COUNT, S_PAGE_RANGE}:
                set_state(context, S_WAIT_FILES)
                return await update.effective_message.reply_text(
                    "به مرحلهٔ ارسال فایل برگشتید.\n"
                    "فایل جدید بفرستید یا «✅ ثبت سفارش» را بزنید.",
                    reply_markup=confirm_kb(),
                )
            if st == S_PAGE_RANGE_CUSTOM:
                set_state(context, S_PAGE_RANGE)
                return await update.effective_message.reply_text(
                    "صفحات مورد نظر:", reply_markup=page_range_kb()
                )
            if st == S_NUP:
                set_state(context, S_PAGE_RANGE)
                return await update.effective_message.reply_text(
                    "صفحات مورد نظر:", reply_markup=page_range_kb()
                )
            if st == S_DUPLEX:
                set_state(context, S_NUP)
                return await update.effective_message.reply_text(
                    "چند صفحه در هر برگ چاپ شود؟", reply_markup=nup_kb()
                )
            if st == S_COPIES:
                set_state(context, S_DUPLEX)
                return await update.effective_message.reply_text(
                    "یک‌رو/دو‌رو:", reply_markup=duplex_kb()
                )
            if st == S_PAPER_SIZE:
                set_state(context, S_COPIES)
                return await update.effective_message.reply_text(
                    "تعداد نسخه:", reply_markup=copies_kb()
                )
            if st == S_PAPER_TYPE:
                set_state(context, S_PAPER_SIZE)
                return await update.effective_message.reply_text(
                    "اندازه کاغذ:", reply_markup=paper_size_kb()
                )
            if st == S_COLOR:
                set_state(context, S_PAPER_TYPE)
                return await update.effective_message.reply_text(
                    "نوع کاغذ:", reply_markup=paper_type_kb()
                )
            if st == S_PAGECOUNT_CONFIRM:
                set_state(context, S_COLOR)
                return await update.effective_message.reply_text(
                    "نوع چاپ را انتخاب کنید:", reply_markup=colour_kb()
                )
            if st == S_CONFIRM:
                set_state(context, S_PAGECOUNT_CONFIRM)
                od = context.user_data.get("order") or {}
                body = _pages_summary(od) + "\n\nتعداد صفحات را تایید می‌کنید؟"
                return await update.effective_message.reply_text(
                    body, reply_markup=pages_confirm_kb()
                )
            if st == S_PAY_METHOD:
                set_state(context, S_CONFIRM)
                return await update.effective_message.reply_text(
                    "بررسی و تایید اطلاعات سفارش:",
                    reply_markup=confirm_kb(),
                )

            # Operator panel
            if st in {
                S_OP_MENU,
                S_OP_ORDERS,
                S_OP_GALLERY,
                S_OP_SEARCH,
                S_OP_SEARCH_QUERY,
            }:
                if st == S_OP_MENU:
                    set_state(context, S_NONE)
                    return await update.effective_message.reply_text(
                        "بازگشت به منوی اصلی.",
                        reply_markup=main_menu_for(uid, context),
                    )
                set_state(context, S_OP_MENU)
                return await update.effective_message.reply_text(
                    "پنل اپراتور:", reply_markup=operator_menu_kb()
                )

            # Admin panel
            if st == S_MG_CAT:
                cat = context.user_data.get("mg_cat")
                if cat is None:
                    set_state(context, S_NONE)
                    return await update.effective_message.reply_text(
                        "بازگشت به منوی اصلی.",
                        reply_markup=main_menu_for(uid, context),
                    )
                context.user_data["mg_cat"] = None
                return await update.effective_message.reply_text(
                    "مدیریت:", reply_markup=admin_root_menu_kb()
                )

            # Tariff wizard
            if st in {
                S_TAR_FILE_KIND,
                S_TAR_SIZE,
                S_TAR_COLOR,
                S_TAR_DUPLEX,
                S_TAR_PTYPE,
                S_TAR_PRICE,
            }:
                context.user_data.pop("tariff", None)
                set_state(context, S_MG_CAT)
                context.user_data["mg_cat"] = "prices"
                return await update.effective_message.reply_text(
                    "به منوی قیمت‌ها برگشتید.", reply_markup=prices_menu_kb()
                )

            return await goto_home(update, context, "بازگشت به منوی اصلی.")

        # ------------------------ global tariffs from anywhere ------------------------
        if st in {S_NONE, S_USER, S_MG_CAT} and wants_show_prices(ntx):
            rows = db.list_tariffs()
            body = (
                "\n".join(
                    [
                        f"- {r['file_kind']} | {r['paper_size']} | {r['color']} | "
                        f"{r['duplex']} | {r['paper_type']} : {r['price_per_sheet']:,}"
                        for r in rows
                    ]
                )
                or "تعرفه‌ای ثبت نشده."
            )
            return await update.effective_message.reply_text(
                body, reply_markup=main_menu_for(uid, context)
            )

        # ------------------------ customer main ------------------------
        if st in {S_NONE, S_USER} and _eq(ntx, L_START_ORDER):
            return await begin_order(update, context)
        if st in {S_NONE, S_USER} and _eq(ntx, L_PRICES):
            rows = db.list_tariffs()
            body = (
                "\n".join(
                    [
                        f"- {r['file_kind']} | {r['paper_size']} | {r['color']} | "
                        f"{r['duplex']} | {r['paper_type']} : {r['price_per_sheet']:,}"
                        for r in rows
                    ]
                )
                or "تعرفه‌ای ثبت نشده."
            )
            return await update.effective_message.reply_text(
                body, reply_markup=main_menu_for(uid, context)
            )
        if st in {S_NONE, S_USER} and _eq(ntx, L_TRACK):
            set_state(context, S_USER)
            return await update.effective_message.reply_text(
                "کد سفارش را ارسال کنید.", reply_markup=rkm([[]])
            )

        # order wizard states
        if st == S_WAIT_FILES:
            return await confirm_or_continue(update, context)
        if st == S_PAGE_COUNT:
            return await handle_page_count(update, context)
        if st == S_PAGE_RANGE:
            return await handle_page_range(update, context)
        if st == S_PAGE_RANGE_CUSTOM:
            return await handle_page_range_custom(update, context)
        if st == S_PAGECOUNT_CONFIRM:
            return await handle_pagecount_confirm(update, context)
        if st == S_NUP:
            return await handle_nup(update, context)
        if st == S_DUPLEX:
            return await handle_duplex(update, context)
        if st == S_COPIES:
            return await handle_copies(update, context)
        if st == S_PAPER_SIZE:
            return await handle_paper_size(update, context)
        if st == S_PAPER_TYPE:
            return await handle_paper_type(update, context)
        if st == S_COLOR:
            return await handle_color(update, context)
        if st == S_CONFIRM and _eq(ntx, "✅ ثبت سفارش"):
            return await finalize_order(update, context)
        if st == S_PAY_METHOD:
            return await handle_pay_method(update, context)

        # ------------------------ operator ------------------------
        if st in {S_NONE, S_USER} and _eq(ntx, L_OPERATOR_PANEL):
            if not is_owner_or_admin(uid, context):
                # فقط اپراتورهای ثبت‌شده / ادمین
                rows = db.list_staff(role="operator")
                if not any(int(r["user_id"]) == int(uid) for r in rows):
                    return await update.effective_message.reply_text(
                        "دسترسی اپراتور برای شما فعال نیست.",
                        reply_markup=main_menu_for(uid, context),
                    )
            return await op_root(update, context)

        if st == S_OP_MENU and _eq(ntx, OP_ORDERS):
            return await op_orders(update, context)
        if st == S_OP_MENU and _eq(ntx, OP_TOGGLE_DUTY):
            rows = db.list_staff(role="operator")
            my = next((r for r in rows if int(r["user_id"]) == int(uid)), None)
            if my is None:
                if is_owner_or_admin(uid, context):
                    db.add_staff(uid, "operator")
                    my = {"user_id": uid, "on_duty": 0}
                else:
                    return await update.effective_message.reply_text(
                        "شما اپراتور نیستید.", reply_markup=operator_menu_kb()
                    )
            new_state = not bool(int(my.get("on_duty") or 0))
            db.set_duty(uid, new_state, "operator")
            status_txt = "🟢 آماده‌به‌کار" if new_state else "🔴 غیرفعال"
            return await update.effective_message.reply_text(
                f"وضعیت شما: {status_txt}", reply_markup=operator_menu_kb()
            )
        if st == S_OP_ORDERS and _eq(ntx, OP_GALLERY):
            set_state(context, S_OP_GALLERY)
            return await gallery_start(update, context)
        if st == S_OP_ORDERS and _eq(ntx, OP_SEARCH):
            set_state(context, S_OP_SEARCH_QUERY)
            return await op_search_start(update, context)
        if st == S_OP_SEARCH_QUERY:
            return await op_search_query(update, context)

        # ------------------------ admin ------------------------
        if (
            st in {S_NONE, S_USER}
            and _eq(ntx, L_ADMIN_PANEL)
            and is_owner_or_admin(uid, context)
        ):
            return await admin_root(update, context)
        if st == S_MG_CAT:
            return await admin_handle(update, context)

        # tariff wizard
        if st == S_TAR_FILE_KIND:
            return await tar_file_kind(update, context)
        if st == S_TAR_SIZE:
            return await tar_size(update, context)
        if st == S_TAR_COLOR:
            return await tar_color(update, context)
        if st == S_TAR_DUPLEX:
            return await tar_duplex(update, context)
        if st == S_TAR_PTYPE:
            return await tar_ptype(update, context)
        if st == S_TAR_PRICE:
            return await tar_price(update, context)

        # tracking by numeric id
        cln = _clean(ntx)
        if re.fullmatch(r"\d{1,12}", cln):
            row = db.get_order(int(cln))
            if not row:
                return await update.effective_message.reply_text(
                    "سفارشی با این کد یافت نشد.",
                    reply_markup=main_menu_for(uid, context),
                )
            body = (
                _build_order_text(row["data"], order_id=row["id"])
                + f"\n\nوضعیت: {row['status']}"
            )
            return await update.effective_message.reply_text(
                body, parse_mode="HTML", reply_markup=main_menu_for(uid, context)
            )

        return await update.effective_message.reply_text(
            "از منو استفاده کنید یا /start را بفرستید.",
            reply_markup=main_menu_for(uid, context),
        )

    # -------------------------------------------------------------- handlers

    app.add_handler(CommandHandler("start", start))
    app.add_handler(CallbackQueryHandler(on_cb))
    app.add_handler(MessageHandler(filters.Document.ALL, on_document))
    app.add_handler(MessageHandler(filters.PHOTO, on_photo))
    app.add_handler(
        MessageHandler(filters.TEXT & ~filters.COMMAND, handle_text), group=10
    )

    app.run_polling()
