from __future__ import annotations

from dataclasses import dataclass, field
from typing import List, Optional
import os


@dataclass
class Settings:
    telegram_bot_token: str
    database_path: str
    admin_chat_id: Optional[int] = None
    owner_user_id: Optional[int] = None
    operator_chat_id: Optional[int] = None
    operator_chat_ids: List[int] = field(default_factory=list)
    big_pdf_mb: int = 100
    timeout_pageops: int = 10


def _get_int(name: str) -> Optional[int]:
    raw = os.getenv(name, "").strip()
    if not raw:
        return None
    try:
        return int(raw)
    except ValueError:
        return None


def load_settings() -> Settings:
    """Read configuration from environment/.env and return a Settings object.

    Expected variables (all optional except TELEGRAM_BOT_TOKEN):

      TELEGRAM_BOT_TOKEN      – bot token from BotFather (required)
      DATABASE_URL / DATABASE_PATH
                               – either a sqlite path (e.g. negarprint.db)
                                 or a MySQL/MariaDB URL of the form:
                                 mysql://user:pass@host:3306/dbname
      ADMIN_CHAT_ID           – chat id of default admin (int)
      OWNER_USER_ID           – telegram user id of technical owner (int)
      OPERATOR_CHAT_ID        – single default operator chat id (int)
      OPERATOR_CHAT_IDS       – comma separated list of operator chat ids
      BIG_PDF_MB              – threshold size (MB) to use longer timeouts
      TIMEOUT_PAGEOPS         – base timeout for reading PDFs/Office files
    """
    # Try to load .env if python-dotenv is available
    try:
        from dotenv import load_dotenv  # type: ignore
        load_dotenv()
    except Exception:
        pass

    token = os.getenv("TELEGRAM_BOT_TOKEN", "").strip()
    if not token:
        raise RuntimeError("TELEGRAM_BOT_TOKEN is not set")

    db_url = (
        os.getenv("DATABASE_URL")
        or os.getenv("DATABASE_PATH")
        or "negarprint.db"
    )

    admin_chat_id = _get_int("ADMIN_CHAT_ID")
    owner_user_id = _get_int("OWNER_USER_ID")
    operator_chat_id = _get_int("OPERATOR_CHAT_ID")

    op_ids_raw = os.getenv("OPERATOR_CHAT_IDS", "")
    operator_chat_ids: List[int] = []
    for part in op_ids_raw.split(","):
        part = part.strip()
        if not part:
            continue
        try:
            operator_chat_ids.append(int(part))
        except ValueError:
            continue

    big_pdf_mb = int(os.getenv("BIG_PDF_MB", "100") or 100)
    timeout_pageops = int(os.getenv("TIMEOUT_PAGEOPS", "10") or 10)

    return Settings(
        telegram_bot_token=token,
        database_path=db_url,
        admin_chat_id=admin_chat_id,
        owner_user_id=owner_user_id,
        operator_chat_id=operator_chat_id,
        operator_chat_ids=operator_chat_ids,
        big_pdf_mb=big_pdf_mb,
        timeout_pageops=timeout_pageops,
    )
