from __future__ import annotations

import time
import sys
import signal
from typing import Dict, Tuple
from subprocess import Popen

from sqlalchemy import select
from sqlalchemy.orm import Session

from app.db.session import SessionLocal
from app.models import Bot as BotModel


def _desired_bots(db: Session) -> Dict[str, Tuple[str, str]]:
    """Return mapping token -> (tenant_id, token) for active store bots."""
    rows = db.scalars(
        select(BotModel).where(BotModel.type == "store", BotModel.is_active == True)  # noqa: E712
    ).all()
    result: Dict[str, Tuple[str, str]] = {}
    for r in rows:
        if not r.tenant_id or not r.token:
            continue
        result[r.token] = (str(r.tenant_id), r.token)
    return result


def main() -> None:
    procs: Dict[str, Popen] = {}
    backoff = 1
    while True:
        try:
            with SessionLocal() as db:
                desired = _desired_bots(db)

            # start new or restarted bots
            for token, (tenant_id, _tok) in desired.items():
                p = procs.get(token)
                if p is None or p.poll() is not None:
                    # spawn a new process for this bot
                    cmd = [
                        sys.executable,
                        "-m",
                        "app.bots.store.serve",
                        "--tenant-id",
                        tenant_id,
                        "--token",
                        token,
                    ]
                    procs[token] = Popen(cmd)

            # stop bots that are no longer desired
            for token, proc in list(procs.items()):
                if token not in desired:
                    try:
                        proc.send_signal(signal.SIGTERM)
                    except Exception:
                        pass
                    procs.pop(token, None)

            # reset backoff on success
            backoff = 1
            time.sleep(15)
        except Exception:
            time.sleep(backoff)
            backoff = min(backoff * 2, 60)


if __name__ == "__main__":
    main()

