from __future__ import annotations
import asyncio
import atexit
from threading import Lock
from typing import Any
from urllib.parse import urlparse, urlunparse
import httpx
from app.core.config import get_settings


settings = get_settings()
_CLIENT_LOCK = Lock()
_SHARED_HTTP_CLIENT: httpx.AsyncClient | None = None
_SHUTDOWN_REGISTERED = False


def _build_http_client() -> httpx.AsyncClient:
    timeout = httpx.Timeout(connect=5.0, read=20.0, write=20.0, pool=5.0)
    limits = httpx.Limits(max_connections=100, max_keepalive_connections=32)
    # shared hosting behind Passenger does not play nicely with HTTP/2, so stick to HTTP/1.1
    return httpx.AsyncClient(timeout=timeout, follow_redirects=True, limits=limits, http2=False)


def _get_http_client() -> httpx.AsyncClient:
    global _SHARED_HTTP_CLIENT, _SHUTDOWN_REGISTERED
    if _SHARED_HTTP_CLIENT is None:
        with _CLIENT_LOCK:
            if _SHARED_HTTP_CLIENT is None:
                _SHARED_HTTP_CLIENT = _build_http_client()
                if not _SHUTDOWN_REGISTERED:
                    atexit.register(_close_http_client)
                    _SHUTDOWN_REGISTERED = True
    return _SHARED_HTTP_CLIENT


def _close_http_client() -> None:
    global _SHARED_HTTP_CLIENT
    client = _SHARED_HTTP_CLIENT
    if client is None:
        return
    try:
        asyncio.run(client.aclose())
    except RuntimeError:
        loop = asyncio.new_event_loop()
        try:
            loop.run_until_complete(client.aclose())
        finally:
            loop.close()
    finally:
        _SHARED_HTTP_CLIENT = None


class BackendClient:
    def __init__(self, token: str | None = None, telegram_id: int | None = None):
        self.token = token
        self.telegram_id = telegram_id
        raw_base = settings.API_BASE_URL.rstrip('/')
        parsed = urlparse(raw_base)
        if parsed.hostname == "localhost":
            host = "127.0.0.1"
            netloc = host if not parsed.port else f"{host}:{parsed.port}"
            parsed = parsed._replace(netloc=netloc)
            raw_base = urlunparse(parsed)
        self.base_url = raw_base.rstrip('/')

    async def login(self, username: str, password: str) -> dict[str, Any]:
        client = _get_http_client()
        try:
            resp = await client.post(
                f"{self.base_url}/auth/login",
                json={"username": username, "password": password},
            )
            resp.raise_for_status()
        except httpx.HTTPStatusError as exc:
            status = exc.response.status_code if exc.response else "unknown"
            detail = exc.response.text if exc.response is not None else str(exc)
            raise ValueError(f"auth failed ({status}): {detail}") from exc
        except httpx.HTTPError as exc:
            raise ConnectionError(f"auth request error: {exc}") from exc
        data = resp.json()
        self.token = data["token"]
        return data

    def _build_headers(self, tenant_id: str | None = None) -> dict[str, str]:
        headers: dict[str, str] = {}
        if self.token:
            headers["Authorization"] = f"Bearer {self.token}"
        if self.telegram_id is not None:
            headers["X-Telegram-User"] = str(self.telegram_id)
        if tenant_id:
            headers["X-Tenant-Id"] = tenant_id
        return headers

    async def post(self, path: str, json: dict[str, Any], tenant_id: str | None = None) -> dict[str, Any]:
        headers = self._build_headers(tenant_id)
        client = _get_http_client()
        resp = await client.post(f"{self.base_url}{path}", headers=headers, json=json)
        resp.raise_for_status()
        return resp.json()

    async def get(self, path: str, tenant_id: str | None = None, params: dict[str, Any] | None = None) -> Any:
        headers = self._build_headers(tenant_id)
        client = _get_http_client()
        resp = await client.get(f"{self.base_url}{path}", headers=headers, params=params)
        resp.raise_for_status()
        return resp.json()

    async def put(self, path: str, json: dict[str, Any] | None = None, tenant_id: str | None = None) -> dict[str, Any]:
        headers = self._build_headers(tenant_id)
        client = _get_http_client()
        resp = await client.put(f"{self.base_url}{path}", headers=headers, json=json or {})
        resp.raise_for_status()
        return resp.json()

    async def delete(self, path: str, tenant_id: str | None = None) -> Any:
        headers = self._build_headers(tenant_id)
        client = _get_http_client()
        resp = await client.delete(f"{self.base_url}{path}", headers=headers)
        if resp.status_code >= 400:
            resp.raise_for_status()
        return resp.json() if resp.content else None
