import os, sys, asyncio, importlib, threading, concurrent.futures
from pathlib import Path
from telegram import Update

# مسیر سورس اصلی پروژه (بستهٔ آماده: src_original)
SRC_ROOT = Path(__file__).resolve().parents[1] / "src_original"
if str(SRC_ROOT) not in sys.path:
    sys.path.insert(0, str(SRC_ROOT))

_APP = None
_LOOP: asyncio.AbstractEventLoop | None = None
_THREAD: threading.Thread | None = None
_STARTED = False
_LOCK = threading.Lock()

def _start_loop_thread():
    global _LOOP, _THREAD
    if _LOOP and _THREAD and _THREAD.is_alive():
        return
    _LOOP = asyncio.new_event_loop()
    def _runner():
        asyncio.set_event_loop(_LOOP)
        _LOOP.run_forever()
    _THREAD = threading.Thread(target=_runner, name="ptb-loop", daemon=True)
    _THREAD.start()

def _run_coro(coro, timeout: float | None = 10.0):
    """Submit coroutine to the global loop and (optionally) wait for result."""
    if not _LOOP:
        raise RuntimeError("Global event loop not initialized")
    fut = asyncio.run_coroutine_threadsafe(coro, _LOOP)
    return fut.result(timeout=timeout) if timeout else fut

def _capture_application():
    """
    bot_ptb.main() را اجرا می‌کنیم، اما run_polling را hook می‌کنیم تا Application را بگیریم
    بدون آن‌که polling واقعا اجرا شود.
    """
    from telegram.ext import Application
    orig = Application.run_polling
    app_ref = {"app": None}

    def _capture(self, *a, **kw):
        app_ref["app"] = self
        return None

    Application.run_polling = _capture
    try:
        mod = importlib.import_module("bot_ptb")
        if not hasattr(mod, "main"):
            raise RuntimeError("bot_ptb.main() not found")
        mod.main()
    finally:
        Application.run_polling = orig

    if not app_ref["app"]:
        raise RuntimeError("Failed to capture PTB Application from bot_ptb.main()")
    return app_ref["app"]

def _ensure_app_started():
    global _APP, _STARTED
    with _LOCK:
        if _APP is None:
            _APP = _capture_application()
        _start_loop_thread()
        if not _STARTED:
            # initialize + start روی همون loop سراسری
            _run_coro(_APP.initialize(), timeout=20.0)
            _run_coro(_APP.start(), timeout=20.0)
            _STARTED = True

def process_update_sync(update_dict: dict, token: str):
    """
    توسط روت وبهوک صدا زده می‌شود. اپ را بالا می‌آورد (اگر لازم باشد)
    و آپدیت را روی loop سراسری پردازش می‌کند.
    """
    _ensure_app_started()
    # مهم: از app.bot استفاده شود؛ Bot جدا نسازیم
    upd = Update.de_json(update_dict, _APP.bot)
    # زمان انتظار کوتاه بگذار تا پاسخ وبهوک سریع باشد؛ خود PTB queue دارد
    try:
        _run_coro(_APP.process_update(upd), timeout=10.0)
    except concurrent.futures.TimeoutError:
        # اگر طول کشید، اشکالی ندارد؛ نگذار وبهوک تایم‌اوت شود
        pass
