import os
import sqlite3
from pathlib import Path
from typing import List, Optional, Tuple

import pandas as pd

_BASE_DIR = Path(__file__).resolve().parents[1]
_DEFAULT_DB = _BASE_DIR / "guarantees.db"
_DB_PATH = Path(os.getenv("DB_PATH") or os.getenv("DATABASE_PATH") or _DEFAULT_DB)
try:
    _DB_PATH.parent.mkdir(parents=True, exist_ok=True)
except Exception:
    pass
DB_PATH = str(_DB_PATH)

def _connect():
    """return sqlite connection with FK support enabled"""
    conn = sqlite3.connect(DB_PATH)
    try:
        conn.execute("PRAGMA foreign_keys = ON")
    except Exception:
        pass
    return conn

def init_db():
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS guarantees (
                code TEXT PRIMARY KEY,
                user_id INTEGER,
                name TEXT,
                mobile TEXT,
                store TEXT,
                photo TEXT,
                issued_at TEXT,
                expires_at TEXT
            )
        """)
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS stores (
                code TEXT PRIMARY KEY,
                name TEXT NOT NULL
            )
        """)
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS catalogs (
                code TEXT PRIMARY KEY,
                title TEXT NOT NULL
            )
        """)
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS catalog_files (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                catalog_code TEXT NOT NULL,
                file_id TEXT NOT NULL,
                file_name TEXT,
                FOREIGN KEY (catalog_code) REFERENCES catalogs(code) ON DELETE CASCADE
            )
        """)
        cursor.execute("CREATE INDEX IF NOT EXISTS idx_catalog_files_code ON catalog_files(catalog_code)")
        conn.commit()

def add_guarantee(data):
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("""
            INSERT INTO guarantees VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        """, (
            data["code"], data["user_id"], data["name"], data["mobile"],
            data["store"], data["photo"], data["issued_at"], data["expires_at"]
        ))
        conn.commit()

def get_by_code(code):
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM guarantees WHERE code=?", (code,))
        return cursor.fetchone()

def search_guarantees(query):
    with _connect() as conn:
        cursor = conn.cursor()
        q = f"%{query}%"
        cursor.execute("""
            SELECT * FROM guarantees
            WHERE name LIKE ? OR mobile LIKE ? OR store LIKE ? OR code LIKE ?
            ORDER BY issued_at DESC
        """, (q, q, q, q))
        return cursor.fetchall()


def get_by_user(user_id):
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "SELECT * FROM guarantees WHERE user_id=? ORDER BY issued_at DESC",
            (user_id,),
        )
        return cursor.fetchall()


def list_all(limit=50):
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "SELECT * FROM guarantees ORDER BY issued_at DESC LIMIT ?",
            (limit,),
        )
        return cursor.fetchall()

def export_all():
    with _connect() as conn:
        return pd.read_sql_query("SELECT * FROM guarantees", conn)

def list_stores() -> List[Tuple[str, str]]:
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("SELECT code, name FROM stores ORDER BY name COLLATE NOCASE")
        return cursor.fetchall()

def get_store_by_code(code: str) -> Optional[Tuple[str, str]]:
    code = (code or "").strip().upper()
    if not code:
        return None
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("SELECT code, name FROM stores WHERE code=?", (code,))
        return cursor.fetchone()

def add_store(code: str, name: str) -> None:
    clean_code = (code or "").strip().upper()
    clean_name = (name or "").strip()
    if not clean_code:
        raise ValueError("Store code is required")
    if not clean_name:
        raise ValueError("Store name is required")
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("INSERT INTO stores(code, name) VALUES (?, ?)", (clean_code, clean_name))
        conn.commit()

def update_store(code: str, new_name: str | None = None, new_code: str | None = None) -> int:
    current_code = (code or "").strip().upper()
    if not current_code:
        return 0
    updates = []
    params = []
    if new_name is not None:
        updates.append("name = ?")
        params.append((new_name or "").strip())
    if new_code is not None:
        updates.append("code = ?")
        params.append((new_code or "").strip().upper())
    if not updates:
        return 0
    params.append(current_code)
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute(f"UPDATE stores SET {', '.join(updates)} WHERE code=?", params)
        conn.commit()
        return cursor.rowcount

def search_stores(query: str, limit: int = 20) -> List[Tuple[str, str]]:
    text = (query or "").strip().upper()
    if not text:
        return []
    pattern = f"%{text}%"
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute(
            """
            SELECT code, name FROM stores
            WHERE UPPER(code) LIKE ? OR UPPER(name) LIKE ?
            ORDER BY name COLLATE NOCASE
            LIMIT ?
            """,
            (pattern, pattern, limit),
        )
        return cursor.fetchall()


# ------------- Catalog management -------------
def add_catalog(code: str, title: str) -> None:
    clean_code = (code or "").strip().upper()
    clean_title = (title or "").strip()
    if not clean_code:
        raise ValueError("Catalog code is required")
    if not clean_title:
        raise ValueError("Catalog title is required")
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("INSERT INTO catalogs(code, title) VALUES (?, ?)", (clean_code, clean_title))
        conn.commit()

def update_catalog(code: str, new_title: str | None = None, new_code: str | None = None) -> int:
    current_code = (code or "").strip().upper()
    if not current_code:
        return 0
    updates = []
    params = []
    if new_title is not None:
        updates.append("title = ?")
        params.append((new_title or "").strip())
    if new_code is not None:
        updates.append("code = ?")
        params.append((new_code or "").strip().upper())
    if not updates:
        return 0
    params.append(current_code)
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute(f"UPDATE catalogs SET {', '.join(updates)} WHERE code=?", params)
        if new_code is not None:
            new_code_clean = (new_code or "").strip().upper()
            cursor.execute(
                "UPDATE catalog_files SET catalog_code=? WHERE catalog_code=?",
                (new_code_clean, current_code),
            )
        conn.commit()
        return cursor.rowcount

def delete_catalog(code: str) -> int:
    clean_code = (code or "").strip().upper()
    if not clean_code:
        return 0
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("DELETE FROM catalogs WHERE code=?", (clean_code,))
        conn.commit()
        return cursor.rowcount

def list_catalogs() -> List[Tuple[str, str, int]]:
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute(
            """
            SELECT c.code, c.title, COUNT(f.id) as file_count
            FROM catalogs c
            LEFT JOIN catalog_files f ON f.catalog_code = c.code
            GROUP BY c.code, c.title
            ORDER BY c.title COLLATE NOCASE
            """
        )
        return cursor.fetchall()

def get_catalog_by_code(code: str) -> Optional[Tuple[str, str]]:
    clean_code = (code or "").strip().upper()
    if not clean_code:
        return None
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("SELECT code, title FROM catalogs WHERE code=?", (clean_code,))
        return cursor.fetchone()

def search_catalogs(query: str, limit: int = 20) -> List[Tuple[str, str, int]]:
    text = (query or "").strip().upper()
    if not text:
        return []
    pattern = f"%{text}%"
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute(
            """
            SELECT c.code, c.title, COUNT(f.id) as file_count
            FROM catalogs c
            LEFT JOIN catalog_files f ON f.catalog_code = c.code
            WHERE UPPER(c.code) LIKE ? OR UPPER(c.title) LIKE ?
            GROUP BY c.code, c.title
            ORDER BY c.title COLLATE NOCASE
            LIMIT ?
            """,
            (pattern, pattern, limit),
        )
        return cursor.fetchall()

def replace_catalog_files(code: str, files: List[Tuple[str, str | None]]) -> None:
    """
    Replace catalog files with given list.
    files: list of (file_id, file_name) tuples. file_name can be None.
    """
    clean_code = (code or "").strip().upper()
    if not clean_code:
        raise ValueError("Catalog code is required")
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute("DELETE FROM catalog_files WHERE catalog_code=?", (clean_code,))
        cursor.executemany(
            "INSERT INTO catalog_files(catalog_code, file_id, file_name) VALUES (?, ?, ?)",
            [(clean_code, fid, fname) for fid, fname in files],
        )
        conn.commit()

def append_catalog_files(code: str, files: List[Tuple[str, str | None]]) -> None:
    clean_code = (code or "").strip().upper()
    if not clean_code:
        raise ValueError("Catalog code is required")
    if not files:
        return
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.executemany(
            "INSERT INTO catalog_files(catalog_code, file_id, file_name) VALUES (?, ?, ?)",
            [(clean_code, fid, fname) for fid, fname in files],
        )
        conn.commit()

def list_catalog_files(code: str) -> List[Tuple[str, str | None]]:
    clean_code = (code or "").strip().upper()
    if not clean_code:
        return []
    with _connect() as conn:
        cursor = conn.cursor()
        cursor.execute(
            "SELECT file_id, file_name FROM catalog_files WHERE catalog_code=? ORDER BY id ASC",
            (clean_code,),
        )
        return cursor.fetchall()
